/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <sys/time.h>

#include "delay.h"
#include "memory.h"

#ifndef lint
static char sccsid[] = "@(#)delay.c	3.4 2/8/99";
#endif


struct delay_state
{
	struct timeval tv;
	int epsilon, n_states;
	int state;
};


struct delay_state *delay_state_from_handle(void *handle)
{
	return (struct delay_state *)handle;
}


static int old_delay_state(void *handle)
{
	struct delay_state *delay_state;
	struct timeval now;

	delay_state = delay_state_from_handle(handle);

	gettimeofday(&now, NULL);
	if (now.tv_usec < delay_state->epsilon)
	{
		now.tv_sec--;
		now.tv_usec += 1000000 - delay_state->epsilon;
	}
	else
		now.tv_usec -= delay_state->epsilon;

	return delay_state->tv.tv_sec < now.tv_sec ||
		delay_state->tv.tv_sec == now.tv_sec &&
		delay_state->tv.tv_usec < now.tv_usec;
}


/*
**	internal and external
*/


void stop_delay(void *handle)
{
	struct delay_state *delay_state;

	delay_state = delay_state_from_handle(handle);

	delay_state->state = -2;
}


/*
**	external only
*/


void update_delay(void *handle)
{
	struct delay_state *delay_state;
	int n;

	delay_state = delay_state_from_handle(handle);

	gettimeofday(&delay_state->tv, NULL);
	if (delay_state->state < -1)
		return;

	if (delay_state->state == -1)
		delay_state->state = 0;

	n = delay_state->n_states;
	if (n && ++delay_state->state > n)
		delay_state->state = 1;
}


void start_delay(void *handle)
{
	struct delay_state *delay_state;

	delay_state = delay_state_from_handle(handle);

	if (old_delay_state(handle))
		delay_state->state = -1;
	else
		delay_state->state = 0;
}


int check_delay(void *handle)
{
	struct delay_state *delay_state;
	int state;

	delay_state = delay_state_from_handle(handle);

	state = delay_state->state;
	if (state < 0)
		return -1;
	else
		return state;
}


void *init_delay_state(int epsilon, int n_states)
{
	struct delay_state *delay_state;

	delay_state = mem_alloc(sizeof *delay_state);

	delay_state->tv.tv_sec = delay_state->tv.tv_usec = 0;
	delay_state->epsilon = epsilon;
	delay_state->n_states = n_states;

	stop_delay((void *)delay_state);
	return (void *)delay_state;
}


void free_delay_state(void *handle)
{
	mem_free(delay_state_from_handle(handle));
}
