/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <netdb.h>
#include <fcntl.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "auth.h"
#include "display.h"
#include "memory.h"
#include "strings.h"

#ifndef lint
static char sccsid[] = "@(#)display.c	4.1 9/9/00";
#endif


static void get_address(char *display_name, struct sockaddr_in *addr)
{
	struct hostent *hent;
	char *p, *host;
	int n;

	addr->sin_family = AF_INET;
	addr->sin_addr.s_addr = htonl(INADDR_ANY);
	addr->sin_port = htons(WELL_KNOWN_PORT_BASE + 0);

	if (display_name == NULL)
		return;

	if ((p = strchr(display_name, ':')) != NULL)
	{
		addr->sin_port = htons(WELL_KNOWN_PORT_BASE + atoi(p + 1));
		n = p - display_name;
		host = mem_alloc(n + 1);
		strncpy(host, display_name, n);
		host[n] = 0;
	}
	else
		host = display_name;

	if (*host == 0)
	{
		char hostname[MAXHOSTNAMELEN];

		if (!gethostname(hostname, sizeof hostname) &&
			(hent = gethostbyname(hostname)) != NULL)
			memcpy(&addr->sin_addr, *hent->h_addr_list,
				sizeof addr->sin_addr);
	}
	else if ((addr->sin_addr.s_addr = inet_addr(host)) != 0xffffffff)
		;
	else if ((hent = gethostbyname(host)) == NULL)
		addr->sin_addr.s_addr = htonl(INADDR_ANY);
	else
		memcpy(&addr->sin_addr, *hent->h_addr_list,
			sizeof addr->sin_addr);

	if (p != NULL)
		mem_free(host);
}


static int create_socket(void)
{
	int sock;

	if ((sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
		return -1;

	if (fcntl(sock, F_SETFD, FD_CLOEXEC))
	{
		close(sock);
		return -1;
	}

	return sock;
}


/*
**	external
*/


char *default_display_name(void)
{
	char *env;

	if ((env = getenv(Server_Default_Env)) != NULL)
		return env;

	return getenv(Server_Fallback_Env);
}


int bind_to_display(char *display_name)
{
	struct sockaddr_in addr;
	int sock, flag;

	get_address(display_name, &addr);
	addr.sin_addr.s_addr = htonl(INADDR_ANY);

	if ((sock = create_socket()) < 0)
		return -1;

	flag = 1;
	if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR,
		(char *)&flag, sizeof flag))
	{
		close(flag);
		return -1;
	}

	if (bind(sock, (struct sockaddr *)&addr, sizeof addr))
	{
		close(sock);
		return -1;
	}

	if (listen(sock, 5))
	{
		close(sock);
		return -1;
	}

	return sock;
}


int connect_to_display(char *display_name, struct display_auth *auth_p)
{
	struct sockaddr_in addr;
	int sock;

	get_address(display_name, &addr);
	if (addr.sin_addr.s_addr == htonl(INADDR_ANY))
		addr.sin_addr.s_addr = inet_addr(Server_Fallback_IP);

	if ((sock = create_socket()) < 0)
		return -1;

	if (connect(sock, (struct sockaddr *)&addr, sizeof addr))
	{
		close(sock);
		return -1;
	}

	if (auth_p != NULL)
	{
		struct sockaddr_in r_addr;
		int addr_len;

		addr_len = sizeof r_addr;
		if (getpeername(sock, (struct sockaddr *)&r_addr, &addr_len))
		{
			close(sock);
			return -1;
		}

		get_auth_by_addr(auth_p, &r_addr);
	}

	return sock;
}


int accept_display_client(int master, struct display_auth *auth_p)
{
	int addr_len;
	struct sockaddr_in addr;
	int sock;

	addr_len = sizeof addr;
	if ((sock = accept(master, (struct sockaddr *)&addr, &addr_len)) < 0)
		return -1;

	if (fcntl(sock, F_SETFD, FD_CLOEXEC))
	{
		close(sock);
		return -1;
	}

	if (auth_p != NULL)
	{
		struct sockaddr_in l_addr;
		int addr_len;

		addr_len = sizeof l_addr;
		if (getsockname(sock, (struct sockaddr *)&l_addr, &addr_len))
		{
			close(sock);
			return -1;
		}

		get_auth_by_addr(auth_p, &l_addr);
	}

	return sock;
}
