/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include "attr.h"
#include "font.h"
#include "fontcache.h"
#include "genfont.h"
#include "keys.h"
#include "ioqueue.h"
#include "memory.h"
#include "strings.h"

#ifndef lint
static char sccsid[] = "@(#)font.c	4.70 9/23/00";
#endif


#define SMALL_CURSOR_SIZE 3

#define Char_Invalid ((unsigned)-1)

#define Flow_Invalid 32767
#define Flow_Is_Invalid(x) (((x) & ~4095) != 0)


struct font_state
{
	Display *display;
	Window main_win, draw_win;
	Window current_focus;
	int do_redraw, ack_pending;
	unsigned long ack_serial;
	int full_w, full_h, small_w, small_h;
	int small_w_base, small_w_scale;
	int small_h_base, small_h_scale;
	int mapped;
	int save_saver;
	int saver_time, saver_int, saver_blank, saver_exp;
};

typedef struct
{
	int x, y, length;
	unsigned *data;
	unsigned char *damaged;
} wordT;


static Pixmap cache_pixmap;
static GC copygc, eventgc, blank_char_gc;
static int cwidth, cheight;
static int font_width, font_height;

static void *genfont_handle;

static unsigned **image_data, **image_saved;
static unsigned char **image_damaged;
static unsigned char **image_rough_done;

static int *damaged_min, *damaged_n;

static int **flow_dx, **flow_dy;
static int *flow_null_row;
static unsigned char **flow_ref;
static int *flow_rows;

static soft_queueT event_sq, redraw_sq;


static font_key_callbackF key_callback;


static struct font_state *font_state_from_handle(void *handle)
{
	return (struct font_state *)handle;
}


/* parent schedules flush */
static Cursor create_cursor(Display *display, Window win)
{
	Pixmap empty;
	XColor black;

	empty = XCreateBitmapFromData(display, win, "", 1, 1);

	black.red = black.green = black.blue = 0;
	black.flags = DoRed | DoGreen | DoBlue;

	return XCreatePixmapCursor(display, empty, empty,
		&black, &black, 0, 0);
}


static void flush_callback(int fd, void *handle)
{
	struct font_state *state;

	state = font_state_from_handle(handle);

	XFlush(state->display);
	set_write_enable(ConnectionNumber(state->display), 0);
}


/* schedules flush and prevents further redraw */
static void schedule_sync(struct font_state *state)
{
	if (!state->ack_pending)
	{
		state->ack_pending = 1;
		state->ack_serial = NextRequest(state->display);
		XCopyArea(state->display, cache_pixmap, cache_pixmap,
			eventgc, 0, 0, 1, 1, -1, -1);
		set_write_enable(ConnectionNumber(state->display), 1);
	}
}


/* schedules flush and doesn't prevent further redraw */
static void schedule_flush(struct font_state *state)
{
	set_write_enable(ConnectionNumber(state->display), 1);
}


static void set_rectangle(int xmin, int ymin,
	int xmax, int ymax, unsigned saved)
{
	int x, y;

	for (y = ymin; y < ymax; y++)
		for (x = xmin; x < xmax; x++)
			image_saved[y][x] = saved;
}


/* sends X requests, parent schedules flush */
static void do_clear_box(struct font_state *state, int x, int y,
	int w, int h, unsigned attr)
{
	switch (attr)
	{
	case 0:
		XFillRectangle(state->display, state->draw_win, blank_char_gc,
			x * font_width, y * font_height,
			w * font_width, h * font_height);
		set_rectangle(x, y, x + w, y + h, ' ');
		break;

	case A_Background:
		XClearArea(state->display, state->draw_win,
			x * font_width, y * font_height,
			w * font_width, h * font_height, False);
		set_rectangle(x, y, x + w, y + h, ' ' | A_Background);
		break;
	}
}


/* child sends X requests, parent schedules flush */
static void do_draw_char(struct font_state *state, int x, int y, unsigned chr)
{
	int width, height;
	unsigned cache_chr;
	int src_x, src_y, src2_x, src2_y;

	if (chr & A_Background)
		chr &= 0xffff | A_Bold | A_Background | A_Narrow;
	else if (chr & A_Small_Cursor)
		chr &= 0xffff | A_Underline | A_Reverse | A_Bold |
			A_Small_Cursor | A_Narrow;
	else
		chr &= 0xffff | A_Cursor | A_Underline | A_Reverse |
			A_Bold | A_Narrow;

	switch (chr)
	{
	case ' ':
	case ' ' | A_Background:
		do_clear_box(state, x, y, 1, 1, chr & A_Background);
		return;
	}

	image_saved[y][x] = chr;

	x *= font_width;
	y *= font_height;

	cache_chr = chr & ~(A_Small_Cursor | A_Narrow);

	width = chr & A_Narrow ? font_width - 1 : font_width;
	height = font_height;

	load_cache(&src_x, &src_y, cache_chr);

	assert(font_height > SMALL_CURSOR_SIZE);

	if (chr & A_Small_Cursor)
	{
		load_cache(&src2_x, &src2_y, cache_chr | A_Cursor);

		XCopyArea(state->display, cache_pixmap, state->draw_win,
			copygc, src_x, src_y, width,
			font_height - SMALL_CURSOR_SIZE, x, y);

		XCopyArea(state->display, cache_pixmap, state->draw_win,
			copygc, src2_x,
			src2_y + font_height - SMALL_CURSOR_SIZE,
			width, height - (font_height - SMALL_CURSOR_SIZE),
			x, y + font_height - SMALL_CURSOR_SIZE);
	}
	else
		XCopyArea(state->display, cache_pixmap, state->draw_win,
			copygc, src_x, src_y, width, height, x, y);

	if (chr & A_Narrow)
		XClearArea(state->display, state->draw_win,
			x + font_width - 1, y, 1, font_height, False);
}


static unsigned rough_attr(int x, int y)
{
	unsigned v;

	if (!image_damaged[y][x] || image_rough_done[y][x])
		return Char_Invalid;

	v = image_data[y][x];
	if (v & A_Background)
		return A_Background;
	else if (v & A_Narrow)
		return A_Narrow;
	else
		return 0;
}


/* child sends X requests, parent schedules flush */
static void rough_redraw(struct font_state *state, int x0, int y0)
{
	unsigned attr;
	int width, height;
	int x, y;

	attr = rough_attr(x0, y0);
	if ((attr & 0xffff) || (attr & A_Narrow))
		return;

	for (width = 1; x0 + width < cwidth &&
		rough_attr(x0 + width, y0) == attr; width++);

	for (height = 1; y0 + height < cheight; height++)
	{
		for (x = x0; x < x0 + width &&
			rough_attr(x, y0 + height) == attr; x++);

		if (x < x0 + width)
			break;
	}

	do_clear_box(state, x0, y0, width, height, attr);

	for (y = y0; y < y0 + height; y++)
		for (x = x0; x < x0 + width; x++)
		{
			image_saved[y][x] = ' ' | attr;
			image_damaged[y][x] = image_data[y][x] != attr;
			image_rough_done[y][x] = 1;
		}
}


static int is_blank(unsigned ch)
{
	return (ch & ~A_Background) == ' ';
}


static int next_word(int *xP, int *yP, int *lengthP)
{
	int x, y, found_blank, length;
	unsigned *image_dataP;

	x = *xP;
	y = *yP;

	if (y < 0)
	{
		x = -1;
		y = 0;
	}
	else if (x < 0)
		x = -1;

	image_dataP = &image_data[y][x + 1];

	found_blank = x < 0; /* the newline or start of screen is a blank */
	while (y < cheight)
	{
		if (++x >= cwidth)
		{
			x = -1;
			y++;
			image_dataP = &image_data[y][x + 1];
			found_blank = 1; /* the newline */
			continue;
		}

		if (is_blank(*image_dataP++))
		{
			found_blank = 1;
			continue;
		}

		if (!found_blank)
			continue;

		for (length = 1; x + length < cwidth &&
			!is_blank(*image_dataP++); length++);

		*xP = x;
		*yP = y;
		*lengthP = length;
		return 1;
	}

	return 0;
}


static int cmp_word(const void *p1, const void *p2)
{
	const wordT *w1, *w2;
	int n1, n2, n;
	int result;

	w1 = p1;
	w2 = p2;

	n1 = w1->length;
	n2 = w2->length;
	n = n1 < n2 ? n1 : n2;

	result = memcmp(w1->data, w2->data, n * sizeof *w1->data);
	if (result)
		return result;

	if (n1 != n2)
		return n1 - n2;

	return memcmp(w1->damaged, w2->damaged, n * sizeof *w1->damaged);
}


static void clear_flow(void)
{
	int y;

	for (y = 0; y < cheight; y++)
	{
		memset(flow_dx[y], 0, cwidth * sizeof **flow_dx);
		memset(flow_dy[y], 0, cwidth * sizeof **flow_dy);
	}
}


static int check_flow_row(int y)
{
	return memcmp(flow_dx[y], flow_null_row,
			cwidth * sizeof *flow_null_row) ||
		memcmp(flow_dy[y], flow_null_row,
			cwidth * sizeof *flow_null_row);
}


/* sends X requests, parent schedules flush */
static void do_flow(struct font_state *state, int x0, int y0, int dx, int dy)
{
	int width, height;
	int x, y;

	/* XXX the direction for searching should depend on the sign of
	   dx and dy */

	flow_ref[y0 + dy][x0 + dx]--;
	for (width = 1; x0 + width < cwidth; width++)
		if (flow_ref[y0][x0 + width] ||
			flow_dx[y0][x0 + width] != dx ||
			flow_dy[y0][x0 + width] != dy)
			break;
		else
			flow_ref[y0 + dy][x0 + width + dx]--;

	for (height = 1; y0 + height < cheight; height++)
	{
		y = y0 + height;
		for (x = x0; x < x0 + width; x++)
			if (flow_ref[y][x] || flow_dx[y][x] != dx ||
				flow_dy[y][x] != dy)
				break;
			else
				flow_ref[y + dy][x + dx]--;

		if (x < x0 + width)
		{
			int xi;

			for (xi = x0; xi < x; xi++)
				flow_ref[y + dy][xi + dx]++;
			break;
		}
	}

	/* XXX should pay attention to GraphicsExpose or other ways
	   of damaged source rectangle */
	XCopyArea(state->display, state->draw_win, state->draw_win, copygc,
		(x0 + dx) * font_width, (y0 + dy) * font_height,
		width * font_width, height * font_height,
		x0 * font_width, y0 * font_height);

	if (dy > 0 || dy == 0 && dx > 0)
		for (y = y0; y < y0 + height; y++)
			for (x = x0; x < x0 + width; x++)
			{
				image_saved[y][x] =
					image_saved[y + dy][x + dx];
				flow_dx[y][x] = flow_dy[y][x] = 0;
			}
	else
		for (y = y0 + height; y-- > y0;)
			for (x = x0 + width; x-- > x0;)
			{
				image_saved[y][x] =
					image_saved[y + dy][x + dx];
				flow_dx[y][x] = flow_dy[y][x] = 0;
			}
}


/* child sends X requests, if allow_output is true; parent schedules flush */
static int resolve_flow(struct font_state *state, int allow_output)
{
	int x, y, i;
	int output_done;

	for (y = 0; y < cheight; y++)
		if (check_flow_row(y))
			break;
		else
			flow_rows[y] = 0;

	if (y >= cheight)
		return 0; /* no action */

	for (i = 0; i < cheight; i++)
		memset(flow_ref[i], 0, cwidth * sizeof **flow_ref);

	for (; y < cheight; y++)
		if (check_flow_row(y))
		{
			int x, dx, dy;
			int *flow_dxP, *flow_dyP;

			flow_dxP = &flow_dx[y][0];
			flow_dyP = &flow_dy[y][0];
			for (x = 0; x < cwidth; x++)
			{
				if (Flow_Is_Invalid(dx = *flow_dxP))
					*flow_dxP = *flow_dyP = 0;
				else if ((dy = *flow_dyP) != 0 || dx != 0)
					flow_ref[y + dy][x + dx]++;
				flow_dxP++;
				flow_dyP++;
			}

			flow_rows[y] = check_flow_row(y);
		}
		else
			flow_rows[y] = 0;

	output_done = 0;

	for (;;)
	{
		for (y = 0; y < cheight && !flow_rows[y]; y++);
		if (y >= cheight)
			break;

		for (; y < cheight; y++)
			if (flow_rows[y])
			{
				int *flow_dxP, *flow_dyP;
				unsigned char *flow_refP;

				flow_dxP = &flow_dx[y][0];
				flow_dyP = &flow_dy[y][0];
				flow_refP = &flow_ref[y][0];
				for (x = 0; x < cwidth; x++,
					flow_dxP++, flow_dyP++, flow_refP++)
				{
					int dx, dy;

					if (*flow_refP)
						continue;

					dx = *flow_dxP;
					dy = *flow_dyP;
					if (dx || dy)
					{
						if (!allow_output)
							return 1;

						do_flow(state, x, y, dx, dy);
						output_done = 1;
					}
				}
			}

		for (y = 0; y < cheight; y++)
			if (flow_rows[y])
				flow_rows[y] = check_flow_row(y);
	}

	return output_done;
}


/* child sends X requests, if allow_output is true; parent schedules flush */
static int redraw_precalc(struct font_state *state, int allow_output)
{
	int something_damaged;
	int y;

	something_damaged = resolve_flow(state, allow_output);
	if (something_damaged && !allow_output)
		return something_damaged;

	for (y = 0; y < cheight; y++)
	{
		memset(image_rough_done[y], 0,
			cwidth * sizeof **image_rough_done);
		memset(image_damaged[y], 0, cwidth * sizeof **image_damaged);

		if (!memcmp(image_data[y], image_saved[y],
			cwidth * sizeof **image_data))
			damaged_min[y] = damaged_n[y] = 0;
		else
		{
			int x, max;

			for (x = 0; x < cwidth &&
				image_data[y][x] == image_saved[y][x]; x++);

			damaged_min[y] = x;

			for (; x < cwidth; x++)
				if (image_data[y][x] != image_saved[y][x])
				{
					max = x;
					image_damaged[y][x] = 1;
				}

			damaged_n[y] = max + 1 - damaged_min[y];
			something_damaged = 1;
		}
	}

	if (!something_damaged)
		state->do_redraw = 0;

	return something_damaged;
}


static int redraw_pending(soft_queueT sq, void *handle)
{
	struct font_state *state;

	state = font_state_from_handle(handle);

	if (state->ack_pending || !state->do_redraw)
		return 0;

	return redraw_precalc(state, 0 /* allow_output */);
}


/* child sends X requests */
static void process_redraw(soft_queueT sq, void *handle)
{
	struct font_state *state;
	int x, y, nx;
	int i, length;
	int n_words;
	wordT *words;

	state = font_state_from_handle(handle);

	if (!redraw_precalc(state, 1 /* allow_output */))
		return;

	for (y = 0; y < cheight; y++)
		for (x = damaged_min[y], nx = damaged_n[y]; nx-- > 0; x++)
			if (image_damaged[y][x] && !image_rough_done[y][x])
				rough_redraw(state, x, y);

	i = 0;
	for (y = 0; y < cheight; y++)
		for (x = damaged_min[y], nx = damaged_n[y]; nx-- > 0; x++)
			if (image_damaged[y][x])
				if (image_data[y][x] & A_Narrow)
					do_draw_char(state, x, y,
						image_data[y][x]);
				else
					i++;

	if (i == 0)
	{
		schedule_sync(state);
		return;
	}

	n_words = 0;
	words = mem_alloc(sizeof *words);

	x = y = -1;
	while (next_word(&x, &y, &length))
	{
		wordT *w;

		n_words++;
		words = mem_realloc(words, n_words * sizeof *words);
		w = &words[n_words - 1];

		w->x = x;
		w->y = y;
		w->length = length;
		w->data = &image_data[y][x];
		w->damaged = &image_damaged[y][x];
	}

	qsort(words, n_words, sizeof *words, cmp_word);

	for (i = 0; i < n_words; i++)
	{
		int dmin, dmax, j;
		int prev_length;
		unsigned *prev_data;

		if (i > 0)
		{
			prev_length = words[i - 1].length;
			prev_data = words[i - 1].data;
		}
		else
			prev_length = 0;

		x = words[i].x;
		y = words[i].y;
		length = words[i].length;

		for (dmin = 0; dmin < length &&
			!image_damaged[y][x + dmin]; dmin++);

		dmax = dmin;

		for (j = dmin; j < length && j < prev_length &&
			image_data[y][x + j] == prev_data[j]; j++)
			if (image_damaged[y][x + j])
				dmax = j + 1;

		if (dmax > dmin)
		{
			int xprev, yprev;
			int src_x, src_y, dest_x, dest_y;

			xprev = words[i - 1].x;
			yprev = words[i - 1].y;

			src_x = (xprev + dmin) * font_width;
			src_y = yprev * font_height;

			dest_x = (x + dmin) * font_width;
			dest_y = y * font_height;

			/* XXX should pay attention to GraphicsExpose or other
			   ways of damaged source rectangle */
			XCopyArea(state->display, state->draw_win,
				state->draw_win, copygc, src_x, src_y,
				(dmax - dmin) * font_width, font_height,
				dest_x, dest_y);

			for (j = dmin; j < dmax; j++)
				image_saved[y][x + j] =
					image_saved[yprev][xprev + j];
		}

		length -= dmax;
		x += dmax;
		while (length-- > 0)
		{
			if (image_damaged[y][x])
				do_draw_char(state, x, y, image_data[y][x]);
			x++;
		}
	}

	mem_free(words);
	schedule_sync(state);
}


static void process_events(soft_queueT sq, void *handle)
{
	struct font_state *state;
	XEvent event;
	int no_work;

	state = font_state_from_handle(handle);

	no_work = !XEventsQueued(state->display, QueuedAlready);

	if (state->ack_pending &&
		LastKnownRequestProcessed(state->display) >= state->ack_serial)
		state->ack_pending = 0;

	if (no_work)
		return;

	while (XEventsQueued(state->display, QueuedAlready))
	{
		XNextEvent(state->display, &event);

		if (event.xany.send_event)
			continue;

		if (event.type == MappingNotify &&
			event.xmapping.request != MappingPointer)
			XRefreshKeyboardMapping(&event.xmapping);

		if (event.type == Expose &&
			event.xany.window == state->draw_win)
		{
			set_rectangle(event.xexpose.x / font_width,
				event.xexpose.y / font_height,
				(event.xexpose.x + event.xexpose.width +
					font_width - 1) / font_width,
				(event.xexpose.y + event.xexpose.height +
					font_height - 1) / font_height,
				Char_Invalid);
			state->do_redraw = 1;
		}

		if (event.type == KeyPress)
		{
			int key, key_n, key_on_root;
			char *key_str;

			key_on_root = event.xany.window ==
				DefaultRootWindow(state->display);
			key = key_of_event(&event.xkey, &key_str, &key_n);
			key_callback(key, key_str, key_n, key_on_root);
		}
	}
}


static int event_pending(soft_queueT sq, void *handle)
{
	struct font_state *state;

	state = font_state_from_handle(handle);
	return XEventsQueued(state->display, QueuedAlready) > 0;
}


static void event_callback(int fd, void *handle)
{
	struct font_state *state;

	state = font_state_from_handle(handle);

	XEventsQueued(state->display, QueuedAfterReading);
}


/* ----- */


void clear_box(void *handle, int x, int y, int w, int h, unsigned attr)
{
	struct font_state *state;
	unsigned ch;
	int i, j;

	state = font_state_from_handle(handle);

	if (attr & ~A_Background)
		return;

	ch = attr | ' ';

	for (j = 0; j < h; j++)
		for (i = 0; i < w; i++)
		{
			image_data[y + j][x + i] = ch;

			/* XXX workaround against ' ' | A_Background
			   clearing */
			if (ch == ' ')
				flow_dx[y + j][x + i] = flow_dy[y + j][x + i] =
					Flow_Invalid;
		}

	state->do_redraw = 1;
}


/* sends X requests */
int map_font(void *handle, int exclusive_lock)
{
	struct font_state *state;

	state = font_state_from_handle(handle);

	XMapRaised(state->display, state->main_win);
	if (XGrabKeyboard(state->display, state->main_win,
			True /* owner events */, GrabModeAsync, GrabModeAsync,
			CurrentTime) != GrabSuccess ||
		exclusive_lock && XGrabPointer(state->display, state->main_win,
			True /* owner events */, 0 /* event mask */,
			GrabModeAsync, GrabModeAsync, None /* confine to */,
			None /* cursor */, CurrentTime) != GrabSuccess)
	{
		XBell(state->display, 100);
		XBell(state->display, 100);
		XUngrabKeyboard(state->display, CurrentTime);
		return 0;
	}

	/* optimize this */
	if (!exclusive_lock)
		XUngrabPointer(state->display, CurrentTime);

	XSetInputFocus(state->display, state->main_win,
		RevertToPointerRoot, CurrentTime);
	state->current_focus = state->main_win;

	XMoveResizeWindow(state->display, state->main_win,
		0, 0, state->full_w, state->full_h);

	/* XXX prevents early redraw */
	schedule_sync(state);

	state->mapped = 1;
	return 1;
}


void unmap_font(void *handle)
{
	struct font_state *state;

	state = font_state_from_handle(handle);

	XUngrabKeyboard(state->display, CurrentTime);
	XUngrabPointer(state->display, CurrentTime);

	if (state->current_focus != PointerRoot)
	{
		XSetInputFocus(state->display, PointerRoot,
			RevertToPointerRoot, CurrentTime);
		state->current_focus = PointerRoot;
	}

	XMoveResizeWindow(state->display, state->main_win,
		state->full_w - state->small_w,
		state->full_h - state->small_h,
		state->small_w, state->small_h);

	schedule_flush(state);
	state->mapped = 0;
}


void font_set_size_hint(void *handle, int width, int height)
{
	struct font_state *state;

	state = font_state_from_handle(handle);

	state->small_w = state->small_w_base + width * state->small_w_scale;
	state->small_h = state->small_h_base + height * state->small_h_scale;

	if (!state->mapped)
		unmap_font(handle);
}


void copy_box(void *handle, int dest_x, int dest_y,
	int src_x, int src_y, int w, int h)
{
	struct font_state *state;
	int x, y, dx, dy;
	int *flow_dx_srcP, *flow_dx_destP;
	int *flow_dy_srcP, *flow_dy_destP;

	state = font_state_from_handle(handle);

	/* delta to the source */
	dx = src_x - dest_x;
	dy = src_y - dest_y;

	if (dx == 0 && dy == 0)
		return;

	if (dest_y < src_y || dest_y == src_y && dest_x < src_x)
		for (y = 0; y < h; y++)
		{
			flow_dx_srcP = &flow_dx[src_y + y][src_x];
			flow_dx_destP = &flow_dx[dest_y + y][dest_x];
			flow_dy_srcP = &flow_dy[src_y + y][src_x];
			flow_dy_destP = &flow_dy[dest_y + y][dest_x];

			memmove(&image_data[dest_y + y][dest_x],
				&image_data[src_y + y][src_x],
				w * sizeof **image_data);

			for (x = 0; x < w; x++)
			{
				int fdx, fdy;

				fdx = *flow_dx_srcP++ + dx;
				fdy = *flow_dy_srcP++ + dy;

				*flow_dx_destP++ = fdx;
				*flow_dy_destP++ = fdy;
			}
		}
	else
		for (y = h; y-- > 0;)
		{
			flow_dx_srcP = &flow_dx[src_y + y][src_x + w];
			flow_dx_destP = &flow_dx[dest_y + y][dest_x + w];
			flow_dy_srcP = &flow_dy[src_y + y][src_x + w];
			flow_dy_destP = &flow_dy[dest_y + y][dest_x + w];

			memmove(&image_data[dest_y + y][dest_x],
				&image_data[src_y + y][src_x],
				w * sizeof **image_data);

			for (x = w; x-- > 0;)
			{
				int fdx, fdy;

				fdx = *--flow_dx_srcP + dx;
				fdy = *--flow_dy_srcP + dy;

				*--flow_dx_destP = fdx;
				*--flow_dy_destP = fdy;
			}
		}

	state->do_redraw = 1;
}


void draw_char(void *handle, int x, int y, unsigned chr)
{
	image_data[y][x] = chr;
	font_state_from_handle(handle)->do_redraw = 1;
}


/* schedules flush */
void *init_font(font_key_callbackF callback, int *cwidth_p, int *cheight_p)
{
	int i;
	Display *display;
	int screen, depth;
	Window root_win, win;
	XWindowAttributes root_attr;
	XSetWindowAttributes attr;
	struct font_state *state;
	unsigned background_pixel, blank_pixel;
	int draw_x, draw_y, draw_w, draw_h;
	XGCValues gc_val;
	int root_width, root_height;

	assert(N_GREY_STEPS == 16);

	state = mem_alloc(sizeof *state);

	key_callback = callback;

	state->save_saver = 0;

	state->display = display = XOpenDisplay(Open_Display_Arg);
	if (display == NULL)
	{
		fprintf(stderr, Open_Display_Err);
		exit(1);
	}

	screen = DefaultScreen(display);
	root_win = DefaultRootWindow(display);
	depth = DefaultDepth(display, screen);

	if (depth < 8)
	{
		fprintf(stderr, Screen_Depth_Err);
		XCloseDisplay(display);
		exit(1);
	}

	state->current_focus = PointerRoot;

	XSelectInput(display, root_win, KeyPressMask);
	XGetWindowAttributes(display, root_win, &root_attr);

	root_width = root_attr.width;
	root_height = root_attr.height;

	if ((genfont_handle = genfont_init(root_width / *cwidth_p,
		&font_width, &font_height)) == NULL)
	{
		font_width = root_width / *cwidth_p;
		font_height = 2 * font_width;
	}

	cwidth = *cwidth_p = root_width / font_width;
	cheight = *cheight_p = root_height / font_height;

	image_data = mem_alloc(cheight * sizeof *image_data);
	image_saved = mem_alloc(cheight * sizeof *image_saved);
	image_damaged = mem_alloc(cheight * sizeof *image_damaged);
	image_rough_done = mem_alloc(cheight * sizeof *image_rough_done);
	flow_dx = mem_alloc(cheight * sizeof *flow_dx);
	flow_dy = mem_alloc(cheight * sizeof *flow_dy);
	flow_ref = mem_alloc(cheight * sizeof *flow_ref);
	for (i = 0; i < cheight; i++)
	{
		int j;

		image_data[i] = mem_alloc(cwidth * sizeof **image_data);
		image_saved[i] = mem_alloc(cwidth * sizeof **image_saved);
		image_damaged[i] = mem_alloc(cwidth * sizeof **image_damaged);
		image_rough_done[i] = mem_alloc(cwidth *
			sizeof **image_rough_done);
		flow_dx[i] = mem_alloc(cwidth * sizeof **flow_dx);
		flow_dy[i] = mem_alloc(cwidth * sizeof **flow_dy);
		flow_ref[i] = mem_alloc(cwidth * sizeof **flow_ref);

		for (j = 0; j < cwidth; j++)
			image_data[i][j] = image_saved[i][j] =
				' ' | A_Background;
	}

	flow_null_row = mem_alloc(cwidth * sizeof *flow_null_row);
	memset(flow_null_row, 0, cwidth * sizeof *flow_null_row);

	clear_flow();

	flow_rows = mem_alloc(cheight * sizeof *flow_rows);

	damaged_min = mem_alloc(cheight * sizeof *damaged_min);
	damaged_n = mem_alloc(cheight * sizeof *damaged_n);

	attr.event_mask = KeyPressMask;
	attr.do_not_propagate_mask = ButtonPressMask | ButtonReleaseMask |
		ButtonMotionMask | PointerMotionMask;
	attr.override_redirect = True;
	state->main_win = win = XCreateWindow(display, root_win, 0, 0,
		root_width, root_height, 0, depth,
		InputOutput, CopyFromParent,
		CWEventMask | CWDontPropagate | CWOverrideRedirect, &attr);

	draw_w = cwidth * font_width;
	draw_h = cheight * font_height;

	draw_x = (root_width - draw_w) / 2;
	draw_y = (root_height - draw_h) / 2;

	attr.event_mask = ExposureMask;
	attr.win_gravity = SouthEastGravity;
	attr.override_redirect = True;
	state->draw_win = XCreateWindow(display, win, draw_x, draw_y,
		draw_w, draw_h, 0, depth, InputOutput, CopyFromParent,
		CWEventMask | CWWinGravity | CWOverrideRedirect, &attr);

	state->small_w_base = root_width - draw_w - draw_x;
	state->small_h_base = root_height - draw_h - draw_y;
	state->small_w_scale = font_width;
	state->small_h_scale = font_height;
	state->small_w = state->small_w_base + state->small_w_scale;
	state->small_h = state->small_h_base + state->small_h_scale;

	state->full_w = root_width;
	state->full_h = root_height;

	XMapWindow(display, state->draw_win);
	state->mapped = 1;

	{
		Atom sticky_prop, no_type;

		no_type = XInternAtom(display, "NULL", False);
		sticky_prop = XInternAtom(display, Wwm_Sticky_Prop, False);
		XChangeProperty(display, state->main_win, sticky_prop,
			no_type, 8, PropModeReplace, (unsigned char *)"", 0);
	}

	cache_pixmap = fontcache_init(display, screen,
		genfont_handle, font_width, font_height);
	get_font_colors(&background_pixel, &blank_pixel);

	gc_val.graphics_exposures = False;
	gc_val.foreground = blank_pixel;
	blank_char_gc = XCreateGC(display, win,
		GCForeground | GCGraphicsExposures, &gc_val);

	gc_val.graphics_exposures = False;
	copygc = XCreateGC(display, win, GCGraphicsExposures, &gc_val);
	eventgc = XCreateGC(display, win, 0, NULL);

	attr.background_pixel = background_pixel;
	XChangeWindowAttributes(display, state->draw_win, CWBackPixel, &attr);

	attr.background_pixel = BlackPixel(display, screen);
	attr.cursor = create_cursor(display, win);
	XChangeWindowAttributes(display, win, CWBackPixel | CWCursor, &attr);

	grab_keys(display, root_win);

	register_io_descriptor(ConnectionNumber(display));
	set_read_handler(ConnectionNumber(display), event_callback, state);
	set_read_enable(ConnectionNumber(display), 1);
	set_write_handler(ConnectionNumber(display), flush_callback, state);

	event_sq = create_soft_queue(event_pending, process_events, state);
	redraw_sq = create_soft_queue(redraw_pending, process_redraw, state);

	state->do_redraw = 0;
	state->ack_pending = 0;

	schedule_flush(state);
	return (void *)state;
}


void exit_font(void *handle)
{
	struct font_state *state;
	int i;

	state = font_state_from_handle(handle);

	deregister_io_descriptor(ConnectionNumber(state->display));
	release_soft_queue(event_sq);
	release_soft_queue(redraw_sq);

	fontcache_exit();

	if (genfont_handle != NULL)
		genfont_close(genfont_handle);

	for (i = 0; i < cheight; i++)
	{
		mem_free(image_data[i]);
		mem_free(image_saved[i]);
		mem_free(image_damaged[i]);
		mem_free(image_rough_done[i]);
		mem_free(flow_dx[i]);
		mem_free(flow_dy[i]);
		mem_free(flow_ref[i]);
	}
	mem_free(image_data);
	mem_free(image_saved);
	mem_free(image_damaged);
	mem_free(image_rough_done);
	mem_free(flow_dx);
	mem_free(flow_dy);
	mem_free(flow_ref);

	mem_free(flow_null_row);
	mem_free(flow_rows);

	XCloseDisplay(state->display);

	mem_free(state);
}


/* sends X requests */
void font_bell(void *handle)
{
	struct font_state *state;

	state = font_state_from_handle(handle);

	XBell(state->display, 0);
	schedule_flush(state);
}


/* sends X requests */
void font_save_screen(void *handle, int timeout)
{
	struct font_state *state;

	state = font_state_from_handle(handle);

	if (!state->save_saver)
	{
		XGetScreenSaver(state->display, &state->saver_time,
			&state->saver_int, &state->saver_blank,
			&state->saver_exp);
		state->save_saver = 1;
	}

	XSetScreenSaver(state->display, timeout, 0,
		PreferBlanking, DontAllowExposures);
	schedule_flush(state);
}


/* sends X requests */
void font_restore_screen(void *handle)
{
	struct font_state *state;

	state = font_state_from_handle(handle);

	if (state->save_saver)
	{
		XSetScreenSaver(state->display, state->saver_time,
			state->saver_int, state->saver_blank,
			state->saver_exp);
		state->save_saver = 0;
		schedule_flush(state);
	}
}
