/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdlib.h>
#include <math.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "attr.h"
#include "fontcache.h"
#include "genfont.h"
#include "memory.h"

#ifndef lint
static char sccsid[] = "@(#)fontcache.c	4.6 9/18/00";
#endif


#define UNDERLINE_HEIGHT 2

#define CACHE_WIDTH 64
#define CACHE_HEIGHT 64

#define CACHE_SIZE (CACHE_WIDTH * CACHE_HEIGHT)

#define CACHE_ID_NONE ((unsigned)-1)

#define CACHE_COLS 128


/* cache_prev[CACHE_SIZE] = LRU */
static int cache_prev[CACHE_SIZE + 1], cache_next[CACHE_SIZE + 1];

static int cache_col_head[CACHE_COLS];
static int cache_col_next[CACHE_SIZE];

/* character and attributes or CACHE_ID_NONE */
static unsigned cache_id[CACHE_SIZE];

static Pixmap cache_pixmap;
static unsigned colors[3][16];
static unsigned black_pixel, white_pixel;

static char **genfont_buffer;
static XImage *genfont_image;
static GC image_gc;

static Display *display;
static int font_width, font_height;

static void *genfont_handle;


static unsigned short text_foreground_color[3] = { 0, 0, 0 };
static unsigned short text_background_color[3] = { 65535, 65535, 65535 };

static unsigned short cursor_foreground_color[3] = { 0, 0, 0 };
static unsigned short cursor_background_color[3] = { 49152, 49152, 65535 };

static unsigned short border_foreground_color[3] = { 65535, 65535, 65535 };
static unsigned short border_background_color[3] = { 16384, 16384, 16384 };


static void draw_dummy_char(void)
{
	int x, y;

	for (y = 0; y < font_height; y++)
		for (x = 0; x < font_width; x++)
		{
			int dx, dy, dist;

			dx = x - (font_width - 1) / 2;
			dy = y - (font_height - 1) / 2;

			if (dx < 0)
				dx = -dx;
			if (dy < 0)
				dy = -dy;

			if (dx > dy)
				dist = dx;
			else
				dist = dy;

			if (dist > (font_width + 1) / 2)
				dist = (font_width + 1) / 2;

			genfont_buffer[y][x] = N_GREY_STEPS - 1 - dist *
				(N_GREY_STEPS - 1) / ((font_width + 1) / 2);
		}
}


static void draw_in_cache(Display *display, int i, unsigned id)
{
	int cache_x, cache_y;
	int ok, gc_index;
	int bold, underline, reverse;
	int x, y;

	cache_x = i % CACHE_WIDTH * font_width;
	cache_y = i / CACHE_WIDTH * font_height;

	bold = (id & A_Bold) != 0;
	underline = (id & A_Underline) && (id & 0xffff) != ' ';
	reverse = (id & A_Reverse) != 0;

	ok = 0;
	if (genfont_handle != NULL && !bold)
		ok = genfont_plain_char(genfont_handle, genfont_buffer,
			id & 0xffff);
	if (genfont_handle != NULL && bold)
		ok = genfont_bold_char(genfont_handle, genfont_buffer,
			id & 0xffff);

	if (!ok)
		draw_dummy_char();

	if (id & A_Background)
		gc_index = 2;
	else if (id & A_Cursor)
		gc_index = 1;
	else
		gc_index = 0;

	for (y = 0; y < font_height; y++)
	{
		int reverse_line, dot;

		reverse_line = reverse ^ (underline &&
			y == font_height - UNDERLINE_HEIGHT);

		for (x = 0; x < font_width; x++)
		{
			dot = genfont_buffer[y][x];
			if (reverse_line)
				dot = 15 - dot;

			XPutPixel(genfont_image, x, y, colors[gc_index][dot]);
		}
	}

	XPutImage(display, cache_pixmap, image_gc, genfont_image,
		0, 0, cache_x, cache_y, font_width, font_height);
}


static void access_cache(int i)
{
	int prev, next;

	prev = cache_prev[i];
	next = cache_next[i];

	cache_next[prev] = next;
	cache_prev[next] = prev;

	prev = CACHE_SIZE;
	next = cache_next[CACHE_SIZE];

	cache_next[prev] = cache_prev[next] = i;
	cache_prev[i] = prev;
	cache_next[i] = next;
}


static unsigned interpolate_primary(unsigned low, unsigned high, int step)
{
	double l, h, x;
	double gamma = 1.2;
	int y;

	l = pow(low / 65535.0, gamma);
	h = pow(high / 65535.0, gamma);

	x = (step * h + (15 - step) * l) / 15;

	y = 65535.0 * pow(x, 1.0 / gamma);
	if (y < 0)
		return 0;
	else if (y > 65535)
		return 65535;
	else
		return y;
}


static int is_darker(unsigned short c1[3], unsigned short c2[3])
{
	unsigned r1, g1, b1;
	unsigned r2, g2, b2;

	r1 = c1[0] >> 8;
	g1 = c1[1] >> 8;
	b1 = c1[2] >> 8;

	r2 = c2[0] >> 8;
	g2 = c2[1] >> 8;
	b2 = c2[2] >> 8;

	return r1*r1 + g1*g1 + b1*b1 < r2*r2 + g2*g2 + b2*b2;
}


static void alloc_color_range(unsigned dest[16], Colormap cmap,
	unsigned short low[3], unsigned short high[3], int inverse)
{
	int i;
	XColor col;

	for (i = 0; i < 16; i++)
	{
		col.red = interpolate_primary(low[0], high[0], i);
		col.green = interpolate_primary(low[1], high[1], i);
		col.blue = interpolate_primary(low[2], high[2], i);

		if (XAllocColor(display, cmap, &col))
			dest[i] = col.pixel;
		else if (inverse ^ (i < 8))
			dest[i] = white_pixel;
		else
			dest[i] = black_pixel;
	}
}


static void alloc_all_colors(Colormap cmap)
{
	int inverse;
	inverse = is_darker(text_background_color, text_foreground_color);

	alloc_color_range(colors[0], cmap,
		text_background_color, text_foreground_color, inverse);
	alloc_color_range(colors[1], cmap,
		cursor_background_color, cursor_foreground_color, !inverse);
	alloc_color_range(colors[2], cmap,
		border_background_color, border_foreground_color, !inverse);
}


/* ----- */


Pixmap fontcache_init(Display *init_display, int screen,
	void *init_genfont_handle, int width, int height)
{
	int i;

	display = init_display;
	genfont_handle = init_genfont_handle;
	font_width = width;
	font_height = height;

	cache_pixmap = XCreatePixmap(display, RootWindow(display, screen),
		CACHE_WIDTH * font_width, CACHE_HEIGHT * font_height,
		DefaultDepth(display, screen));
	genfont_image = XGetImage(display, cache_pixmap,
		0, 0, font_width, font_height, 0xffffffff, ZPixmap);

	genfont_buffer = mem_alloc(font_height * sizeof *genfont_buffer);
	for (i = 0; i < font_height; i++)
		genfont_buffer[i] = mem_alloc(font_width);

	for (i = 0; i < CACHE_SIZE; i++)
	{
		cache_prev[i + 1] = i;
		cache_next[i] = i + 1;
		cache_id[i] = CACHE_ID_NONE;
	}

	cache_prev[0] = CACHE_SIZE;
	cache_next[CACHE_SIZE] = 0;

	for (i = 0; i < CACHE_COLS; i++)
		cache_col_head[i] = -1;

	image_gc = DefaultGC(display, screen);

	black_pixel = BlackPixel(display, screen);
	white_pixel = WhitePixel(display, screen);

	alloc_all_colors(DefaultColormap(display, screen));

	return cache_pixmap;
}


void get_font_colors(unsigned *background_pixelP, unsigned *blank_pixelP)
{
	*blank_pixelP = colors[0][0];
	*background_pixelP = colors[2][0];
}


void fontcache_exit(void)
{
	int i;

	for (i = 0; i < font_height; i++)
		mem_free(genfont_buffer[i]);
	mem_free(genfont_buffer);

	XDestroyImage(genfont_image);
	XFreePixmap(display, cache_pixmap);

	display = NULL;
}


void load_cache(int *xP, int *yP, unsigned id)
{
	int i, col;
	unsigned evict_id;

	col = id % CACHE_COLS;

	for (i = cache_col_head[col]; i >= 0; i = cache_col_next[i])
		if (cache_id[i] == id)
		{
			access_cache(i);
			*xP = i % CACHE_WIDTH * font_width;
			*yP = i / CACHE_WIDTH * font_height;
			return;
		}

	i = cache_prev[CACHE_SIZE];
	evict_id = cache_id[i];

	if (evict_id != CACHE_ID_NONE)
	{
		int evict_col;
		int j, prev_j;

		evict_col = evict_id % CACHE_COLS;
		j = cache_col_head[evict_col];

		if (j == i)
			cache_col_head[evict_col] = cache_col_next[j];
		else
		{
			while (j != i)
			{
				prev_j = j;
				j = cache_col_next[j];
			}

			cache_col_next[prev_j] = cache_col_next[j];
		}
	}

	cache_col_next[i] = cache_col_head[col];
	cache_col_head[col] = i;

	cache_id[i] = id;
	draw_in_cache(display, i, id);

	access_cache(i);
	*xP = i % CACHE_WIDTH * font_width;
	*yP = i / CACHE_WIDTH * font_height;
}
