/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>

#include "genfont.h"
#include "memory.h"
#include "strings.h"

#ifndef lint
static char sccsid[] = "@(#)genfont.c	4.2 9/11/00";
#endif


#define ORIG_FONT_SIZE ((orig_width + 7) / 8 * orig_height)
#define FONTFILE_SEEK(i) (2 + (ORIG_FONT_SIZE + 2) * (i))

#define BOLD_WIDTH 3
#define BOLD_HEIGHT 3

#define CACHE_SIZE 256


typedef enum { Cache_Unused, Cache_Plain, Cache_Bold } cache_flagE;

typedef struct
{
	cache_flagE flag;
	unsigned id; /* character index; 0 - 255 for 8859-1 */
	char **font_data; /* NULL for Cache_Unused; no_char for file misses */
} cache_entryT;


/* index 0 is LRU */
static cache_entryT font_cache[CACHE_SIZE];

static int font_width, font_height;

static int fontfile_fd, fontfile_entries;
static int orig_width, orig_height;
static unsigned char *fontfile_buffer;
static char **orig_char;

static int x_scale_enum, x_offset_enum, x_denom;
static int y_scale_enum, y_offset_enum, y_denom;

static char *no_char_dummy, **no_char = &no_char_dummy;


static char **alloc_char(int width, int height)
{
	char **data;
	int y;

	data = mem_alloc(height * sizeof *data);
	for (y = 0; y < height; y++)
		data[y] = mem_alloc(width);

	return data;
}


static void free_char(char **data, int width, int height)
{
	int y;

	for (y = 0; y < height; y++)
		mem_free(data[y]);
	mem_free(data);
}


static int scale_pixel(int new_scale_x, int new_scale_y, char **old_char,
	int old_width, int old_height, int old_scale_x, int old_scale_y,
	int px_by_old_scale, int py_by_old_scale)
{
	int sum, x, y;
	int x_min, y_min, x_max, y_max;

	x_min = px_by_old_scale < 0 ? 0 : px_by_old_scale / new_scale_x;
	y_min = py_by_old_scale < 0 ? 0 : py_by_old_scale / new_scale_y;

	x_max = px_by_old_scale + old_scale_x <= 0 ? -1 :
		(px_by_old_scale + old_scale_x - 1) / new_scale_x;
	y_max = py_by_old_scale + old_scale_y <= 0 ? -1 :
		(py_by_old_scale + old_scale_y - 1) / new_scale_y;

	if (x_max >= old_width)
		x_max = old_width - 1;
	if (y_max >= old_height)
		y_max = old_height - 1;

	sum = 0;
	for (x = x_min; x <= x_max; x++)
		for (y = y_min; y <= y_max; y++)
		{
			int xl, xh, yl, yh;

			if (!old_char[y][x])
				continue;

			xl = x * new_scale_x;
			if (xl < px_by_old_scale)
				xl = px_by_old_scale;
			xh = (x + 1) * new_scale_x;
			if (xh > px_by_old_scale + old_scale_x)
				xh = px_by_old_scale + old_scale_x;

			yl = y * new_scale_y;
			if (yl < py_by_old_scale)
				yl = py_by_old_scale;
			yh = (y + 1) * new_scale_y;
			if (yh > py_by_old_scale + old_scale_y)
				yh = py_by_old_scale + old_scale_y;

			sum += (xh - xl) * (yh - yl);
		}

	return sum * N_GREY_STEPS / (old_scale_x * old_scale_y + 1);
}


static void scale_char(char **new_char, char **old_char)
{
	int x, y, scaled_x, scaled_y;

	for (x = 0, scaled_x = -x_offset_enum; x < font_width;
		x++, scaled_x += x_denom)
		for (y = 0, scaled_y = -y_offset_enum; y < font_height;
			y++, scaled_y += y_denom)
			new_char[y][x] = scale_pixel(x_scale_enum,
				y_scale_enum, old_char,
				orig_width, orig_height, x_denom, y_denom,
				scaled_x, scaled_y);
}


static int entry_number(int i)
{
	unsigned char ch_buffer[2];

	if (lseek(fontfile_fd, (off_t)FONTFILE_SEEK(i), SEEK_SET) < 0 ||
		read(fontfile_fd, ch_buffer, 2) != 2)
		return -1;

	return (int)ch_buffer[0] * 256 + (int)ch_buffer[1];
}


static char **load_char(int ch, int bold)
{
	int i, min, max, i_ch;
	char **curr_char;

	min = 0;
	max = fontfile_entries;

	i_ch = entry_number(0);

	while (i_ch != ch && min < max - 1)
	{
		i = (min + max) / 2;
		i_ch = entry_number(i);

		if (i_ch < ch)
			min = i;
		else if (i_ch > ch)
			max = i;
	}

	if (i_ch == ch)
	{
		int w, x, xh, y;
		unsigned char *buffer_p;

		if (read(fontfile_fd, fontfile_buffer,
			ORIG_FONT_SIZE) != ORIG_FONT_SIZE)
			return no_char;

		curr_char = alloc_char(font_width, font_height);

		buffer_p = fontfile_buffer;

		for (y = 0; y < orig_height; y++)
			for (xh = 0; xh < orig_width; xh += 8)
			{
				w = *buffer_p++;

				for (x = 0; x < 8 && xh + x < orig_width; x++)
					orig_char[y][xh + x] = (w >> x) & 1;
			}

		if (bold)
			for (y = 0; y < orig_height; y++)
				for (x = w = 0; x < orig_width; x++)
				{
					if (orig_char[y][x])
						w = BOLD_WIDTH;
					if (w)
					{
						w--;
						orig_char[y][x] = 1;
						if (y)
							orig_char[y - 1][x] = 1;
						if (y > 1)
							orig_char[y - 2][x] = 1;
					}
				}

		scale_char(curr_char, orig_char);
		return curr_char;
	}

	return no_char;
}


static int open_fontfile(void)
{
	int i;
	char *font_name;
	unsigned char ch_buffer;
	off_t size;

	for (i = 0; i < CACHE_SIZE; i++)
	{
		font_cache[i].flag = Cache_Unused;
		font_cache[i].font_data = NULL;
	}

	if ((font_name = getenv(Font_File_Env)) == NULL)
		font_name = Font_File_Fallback_Name;

	if ((fontfile_fd = open(font_name, O_RDONLY, 0)) < 0)
		return 0;

	fcntl(fontfile_fd, F_SETFD, FD_CLOEXEC);

	if (read(fontfile_fd, &ch_buffer, 1) != 1)
	{
		close(fontfile_fd);
		return 0;
	}
	orig_width = ch_buffer;

	if (read(fontfile_fd, &ch_buffer, 1) != 1)
	{
		close(fontfile_fd);
		return 0;
	}
	orig_height = ch_buffer;

	if ((size = lseek(fontfile_fd, (off_t)0, SEEK_END)) < 2)
	{
		close(fontfile_fd);
		return 0;
	}
	fontfile_entries = (size - 2) / (ORIG_FONT_SIZE + 2);

	if (fontfile_entries < 1)
	{
		close(fontfile_fd);
		return 0;
	}

	orig_char = alloc_char(orig_width, orig_height);
	fontfile_buffer = mem_alloc(ORIG_FONT_SIZE);

	return 1;
}


static void init_font_size(int desired_width, int orig_width, int orig_height)
{
	char *filename;
	FILE *f;
	int ow, oh, w, h, xo, xs, xd, yo, ys, yd;

	font_width = desired_width;
	font_height = (desired_width * orig_height + (orig_height - 1)) /
		orig_width;

	x_scale_enum = y_scale_enum = font_width;
	x_offset_enum = y_offset_enum = 0;
	x_denom = y_denom = orig_width;

	if ((filename = getenv(Font_Conf_File_Env)) == NULL)
		filename = Font_Conf_File_Fallback_Name;

	if ((f = fopen(filename, "r")) == NULL)
		return;

	while (fscanf(f, "%d %d %d %d %d %d %d %d %d %d",
		&ow, &oh, &w, &h, &xo, &xs, &xd, &yo, &ys, &yd) == 10)
		if (ow == orig_width && oh == orig_height &&
			w == desired_width)
		{
			font_width = w;
			font_height = h;
			x_offset_enum = xo;
			x_scale_enum = xs;
			x_denom = xd;
			y_offset_enum = yo;
			y_scale_enum = ys;
			y_denom = yd;

			fclose(f);
			return;
		}

	fclose(f);
}


static int fetch_lru_char(char **buffer)
{
	int x, y;

	if (font_cache[0].font_data == no_char)
		return 0;

	for (y = 0; y < font_height; y++)
		for (x = 0; x < font_width; x++)
			buffer[y][x] = font_cache[0].font_data[y][x];
	return 1;
}


static int gen_font_char(unsigned ch, char **buffer, int bold)
{
	int i;

	if ((ch & 0xffff) != ch)
		return 0;

	for (i = 0; i < CACHE_SIZE && font_cache[i].flag != Cache_Unused; i++)
	{
		if (font_cache[i].id != ch)
			continue;

		if (!bold && font_cache[i].flag != Cache_Plain)
			continue;
		if (bold && font_cache[i].flag != Cache_Bold)
			continue;

		if (i > 0)
		{
			cache_entryT c;

			memcpy(&c, &font_cache[i], sizeof c);
			memmove(&font_cache[1], &font_cache[0],
				i * sizeof *font_cache);
			memcpy(&font_cache[0], &c, sizeof c);
		}

		return fetch_lru_char(buffer);
	}

	if (font_cache[CACHE_SIZE - 1].flag != Cache_Unused &&
		font_cache[CACHE_SIZE - 1].font_data != no_char)
		free_char(font_cache[CACHE_SIZE - 1].font_data,
			font_width, font_height);

	memmove(&font_cache[1], &font_cache[0],
		(CACHE_SIZE - 1) * sizeof *font_cache);

	font_cache[0].flag = bold ? Cache_Bold : Cache_Plain;
	font_cache[0].id = ch;
	font_cache[0].font_data = load_char(ch, bold);

	return fetch_lru_char(buffer);
}


static void close_fontfile(void)
{
	close(fontfile_fd);

	free_char(orig_char, orig_width, orig_height);
	mem_free(fontfile_buffer);
}


/*
**	external only
*/


int genfont_plain_char(void *handle, char **buffer, unsigned ch)
{
	return gen_font_char(ch, buffer, 0);
}


int genfont_bold_char(void *handle, char **buffer, unsigned ch)
{
	return gen_font_char(ch, buffer, 1);
}


void *genfont_init(int desired_width, int *font_width_p, int *font_height_p)
{
	if (!open_fontfile())
		return NULL;

	init_font_size(desired_width, orig_width, orig_height);

	if (font_width < 1 || font_height < 1 ||
		orig_width < font_width || orig_height < font_height)
	{
		close_fontfile();
		return NULL;
	}

	*font_width_p = font_width;
	*font_height_p = font_height;

	return ""; /* dummy handle */
}


void genfont_close(void *handle)
{
	close_fontfile();
}
