/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>

#include "ioqueue.h"
#include "memory.h"

#ifndef lint
static char sccsid[] = "@(#)ioqueue.c	4.14 3/2/99";
#endif


struct soft_queueS
{
	io_sq_pendingF pending;
	io_sq_processF process;
	void *handle;
};

typedef struct
{
	int fd; /* used as index or -1 */
	soft_queueT sq; /* used as index or NULL */

	/* only for fd type requests */
	io_callbackF read_callback, write_callback;
	void *read_handle, *write_handle;

	int read_wait, write_wait;
	int try_read, try_write;

	int read_done, write_done;
} io_requestT;


static int n_io_requests = 0;
static io_requestT *io_requests;


extern int errno;


static void push_back_req(int i)
{
	io_requestT r;

	if (i + 1 >= n_io_requests)
		/* last or not found */
		return;

	memcpy(&r, &io_requests[i], sizeof r);
	memmove(&io_requests[i], &io_requests[i + 1],
		(n_io_requests - (i + 1)) * sizeof *io_requests);
	memcpy(&io_requests[n_io_requests - 1], &r, sizeof r);
}


static void push_back(int fd)
{
	int i;

	for (i = 0; i < n_io_requests && io_requests[i].fd != fd; i++);
	push_back_req(i);
}


static void push_back_sq(soft_queueT sq)
{
	int i;

	for (i = 0; i < n_io_requests && io_requests[i].sq != sq; i++);
	push_back_req(i);
}


static io_requestT *find_fd(int fd)
{
	int i;

	if (fd < 0)
		abort();

	for (i = 0; i < n_io_requests; i++)
		if (io_requests[i].fd == fd)
			return &io_requests[i];

	return NULL;
}


static int process_io_request(void)
{
	int i;

	for (i = 0; i < n_io_requests; i++)
	{
		io_requestT *r;

		r = &io_requests[i];

		if (r->sq != NULL && !r->read_done &&
			r->sq->pending(r->sq, r->sq->handle))
		{
			soft_queueT sq;

			r->read_done = 1;

			sq = r->sq;
			push_back_sq(sq);
			sq->process(sq, sq->handle);
			return 1;
		}
		else if (r->fd >= 0 && !r->write_done && r->try_write)
		{
			int fd;
			io_callbackF f;
			void *handle;

			fd = r->fd;
			f = r->write_callback;
			handle = r->write_handle;

			r->write_done = 1;

			push_back(fd);
			f(fd, handle);
			return 1;
		}
		else if (r->fd >= 0 && !r->read_done && r->try_read)
		{
			int fd;
			io_callbackF f;
			void *handle;

			fd = r->fd;
			f = r->read_callback;
			handle = r->read_handle;

			r->read_done = 1;

			push_back(fd);
			f(fd, handle);
			return 1;
		}
	}

	return 0;
}


/* ----- */


void set_read_handler(int fd, io_callbackF f, void *handle)
{
	io_requestT *r;

	if ((r = find_fd(fd)) != NULL)
	{
		r->read_callback = f;
		r->read_handle = handle;
		r->read_wait = r->try_read = 0;
	}
}


void set_write_handler(int fd, io_callbackF f, void *handle)
{
	io_requestT *r;

	if ((r = find_fd(fd)) != NULL)
	{
		r->write_callback = f;
		r->write_handle = handle;
		r->write_wait = r->try_write = 0;
	}
}


void set_read_enable(int fd, int enable)
{
	io_requestT *r;

	if ((r = find_fd(fd)) != NULL && r->read_wait != enable)
	{
		if (enable && r->read_callback == NULL)
			abort();

		r->read_wait = enable;
	}
}


void set_write_enable(int fd, int enable)
{
	io_requestT *r;

	if ((r = find_fd(fd)) != NULL && r->write_wait != enable)
	{
		if (enable && r->write_callback == NULL)
			abort();

		r->write_wait = enable;
	}
}


void register_io_descriptor(int fd)
{
	io_requestT *r;

	if (fd < 0)
		abort();

	if (n_io_requests++)
		io_requests = mem_realloc(io_requests,
			n_io_requests * sizeof *io_requests);
	else
		io_requests = mem_alloc(sizeof *io_requests);

	r = &io_requests[n_io_requests - 1];

	if (fcntl(fd, F_SETFL, O_NONBLOCK))
	{
		perror("fcntl");
		exit(1);
	}

	r->fd = fd;
	r->sq = NULL;
	r->read_callback = NULL;
	r->read_handle = NULL;
	r->write_callback = NULL;
	r->write_handle = NULL;
	r->read_wait = r->try_read = 0;
	r->write_wait = r->try_write = 0;
	r->read_done = r->write_done = 0;
}


void deregister_io_descriptor(int fd)
{
	int i;

	/* don't disable nonblocking because fd might already be closed */

	if (fd < 0)
		abort();

	for (i = 0; i < n_io_requests && io_requests[i].fd != fd; i++);

	if (i >= n_io_requests)
		/* not registered */
		return;

	if (i + 1 < n_io_requests)
		memmove(&io_requests[i], &io_requests[i + 1],
			(n_io_requests - (i + 1)) * sizeof *io_requests);

	if (!--n_io_requests)
		mem_free(io_requests);
}


soft_queueT create_soft_queue(io_sq_pendingF pending,
	io_sq_processF process, void *handle)
{
	soft_queueT sq;
	io_requestT *r;

	sq = mem_alloc(sizeof *sq);

	sq->pending = pending;
	sq->process = process;
	sq->handle = handle;

	if (n_io_requests++)
		io_requests = mem_realloc(io_requests,
			n_io_requests * sizeof *io_requests);
	else
		io_requests = mem_alloc(sizeof *io_requests);

	r = &io_requests[n_io_requests - 1];
	r->fd = -1;
	r->sq = sq;

	return sq;
}


void release_soft_queue(soft_queueT sq)
{
	int i;

	if (sq == NULL)
		abort();

	for (i = 0; i < n_io_requests && io_requests[i].sq != sq; i++);

	if (i >= n_io_requests)
		/* not registered */
		abort();

	if (i + 1 < n_io_requests)
		memmove(&io_requests[i], &io_requests[i + 1],
			(n_io_requests - (i + 1)) * sizeof *io_requests);

	mem_free(sq);
	if (!--n_io_requests)
		mem_free(io_requests);
}


void do_select(void)
{
	struct timeval timeout, *timeoutP;
	fd_set rfds, wfds;
	int i;

	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
	timeoutP = NULL;

	for (i = 0; i < n_io_requests; i++)
		io_requests[i].read_done = io_requests[i].write_done = 0;

	while (process_io_request())
		timeoutP = &timeout;

	FD_ZERO(&rfds);
	FD_ZERO(&wfds);

	for (i = 0; i < n_io_requests; i++)
	{
		io_requestT *r;

		r = &io_requests[i];

		if (r->fd < 0)
			continue;

		if (r->read_wait)
			FD_SET(r->fd, &rfds);

		if (r->write_wait)
			FD_SET(r->fd, &wfds);

		r->try_read = r->try_write = 0;
	}

	switch (select(FD_SETSIZE, &rfds, &wfds, NULL, timeoutP))
	{
	case -1:
		if (errno != EINTR)
		{
			perror("select");
			exit(1);
		}
		return;

	case 0:
		return;
	}

	for (i = 0; i < n_io_requests; i++)
	{
		io_requestT *r;

		r = &io_requests[i];

		if (r->fd >= 0)
		{
			r->try_read = r->read_wait && FD_ISSET(r->fd, &rfds);
			r->try_write = r->write_wait && FD_ISSET(r->fd, &wfds);
		}
	}
}
