/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>

#include "tty.h"
#include "terminal.h"
#include "proto.h"
#include "auth.h"
#include "display.h"
#include "genargv.h"
#include "memory.h"
#include "strings.h"
#include "priv.h"

#ifndef lint
static char sccsid[] = "@(#)jterm.c	4.2 9/23/00";
#endif


#define XN 80
#define YN 25


extern int errno;

static char *net_buf;
static int n_net_buf;
static void *tty_state;
static void *term_state;

static void *callback_handle;


static void shell_input(int ch)
{
	char c;

	if (ch < 1 || ch > 255)
		return;
	c = ch;
	write(fd_of_shell(tty_state), &c, 1);
}


static void parse_buf(char **buf, int *n)
{
	int type, ev_len, i;

	if ((type = get_type_from_buffer((void **)buf, n)) < 0)
		return;

	ev_len = get_len_from_buffer((void **)buf, n);

	if (type == Proto_Ev_Key)
		for (i = 0; i < ev_len; i++)
			shell_input(get_word_from_buffer((void **)buf, n, i));
	else if (type == Proto_Ev_Size && ev_len == 1)
	{
		int w, h;

		get_pair_from_buffer((void **)buf, n, &w, &h, 0);

		resize_term(term_state, w, h);
		resize_shell(tty_state, w, h);
	}

	remove_block_from_buffer((void **)buf, n);
}


static void print_nothing(void)
{
}


/*
**	external
*/


void callback_bell(void *handle)
{
	assert(handle == callback_handle);

	append_header_to_buffer((void **)&net_buf, &n_net_buf,
		Proto_Dreq_Bell, 0);
}


void callback_move_cursor(void *handle, int x, int y)
{
	assert(handle == callback_handle);

	append_header_to_buffer((void **)&net_buf, &n_net_buf,
		Proto_Dreq_Cursor, 1);
	append_pair_to_buffer((void **)&net_buf, &n_net_buf, x & 255, y & 255);
}


void callback_set_char(void *handle, int x, int y, int ch)
{
	assert(handle == callback_handle);

	append_header_to_buffer((void **)&net_buf, &n_net_buf,
		Proto_Dreq_Text, 2);
	append_pair_to_buffer((void **)&net_buf, &n_net_buf, x & 255, y & 255);
	append_word_to_buffer((void **)&net_buf, &n_net_buf, ch);
}


void callback_clear_area(void *handle, int x1, int y1, int x2, int y2)
{
	assert(handle == callback_handle);

	append_header_to_buffer((void **)&net_buf, &n_net_buf,
		Proto_Dreq_Clear_Area, 2);
	append_pair_to_buffer((void **)&net_buf, &n_net_buf,
		x1 & 255, y1 & 255);
	append_pair_to_buffer((void **)&net_buf, &n_net_buf,
		x2 & 255, y2 & 255);
}


void callback_copy_area(void *handle, int x1, int y1,
	int x2, int y2, int x3, int y3)
{
	assert(handle == callback_handle);

	append_header_to_buffer((void **)&net_buf, &n_net_buf,
		Proto_Dreq_Copy_Area, 3);
	append_pair_to_buffer((void **)&net_buf, &n_net_buf,
		x1 & 255, y1 & 255);
	append_pair_to_buffer((void **)&net_buf, &n_net_buf,
		x2 & 255, y2 & 255);
	append_pair_to_buffer((void **)&net_buf, &n_net_buf,
		x3 & 255, y3 & 255);
}


int main(int argc, char **argv)
{
	char *buf;
	char **spawn, *spawn_path, *arg0;
	int n_buf, net_fd;
	struct display_auth auth;
	int login_shell;

	init_and_lower_priv();

	if ((net_fd = connect_to_display(default_display_name(), &auth)) < 0)
	{
		fprintf(stderr, Connect_Err);
		exit(1);
	}

	if ((term_state = init_term_state(callback_handle = (void *)&net_buf,
		XN, YN)) == NULL)
	{
		fprintf(stderr, Init_Term_Emul_Err);
		exit(1);
	}

	if (login_shell = argc > 1 && !strcmp(argv[1], As_Login_Shell_Opt))
		argc--,
		argv++;

	if (argc < 2)
		spawn = gen_argv(Shell_Env, Default_Shell);
	else
		spawn = copy_argv(argv + 1);

	spawn_path = spawn[0];

	if ((arg0 = strrchr(spawn[0], '/')) == NULL)
		arg0 = spawn[0];
	else
		arg0++;

	{
		char *tmp_arg0;

		if (login_shell)
		{
			tmp_arg0 = mem_alloc(strlen(arg0) + 2);
			strcpy(tmp_arg0, "-");
			strcat(tmp_arg0, arg0);
			arg0 = tmp_arg0;
		}
		else
		{
			tmp_arg0 = mem_alloc(strlen(arg0) + 1);
			strcpy(tmp_arg0, arg0);
			arg0 = tmp_arg0;
		}
	}

	spawn[0] = arg0;

	{
		struct sigaction dfl_sa;

		dfl_sa.sa_handler = SIG_DFL;
		sigemptyset(&dfl_sa.sa_mask);
		dfl_sa.sa_flags = 0;

		if (sigaction(SIGINT, &dfl_sa, NULL) ||
			sigaction(SIGQUIT, &dfl_sa, NULL))
		{
			fprintf(stderr, Reset_Sig_Handlers_Err);
			exit(1);
		}
	}

	if ((tty_state = fork_shell(XN, YN, (print_callback_f)print_nothing,
		spawn_path, spawn)) == NULL)
	{
		fprintf(stderr, Fork_Shell_Err);
		exit(1);
	}

	mem_free(spawn_path);
	free_argv(spawn);

	n_buf = n_net_buf = 0;
	if (auth.length)
	{
		append_raw_to_buffer((void **)&net_buf, &n_net_buf,
			Proto_Ireq_Authenticate, auth.length, auth.cookie);
		mem_free(auth.cookie);
	}

	append_header_to_buffer((void **)&net_buf, &n_net_buf,
		Proto_Ireq_Term, 0);

	for (;;)
	{
		fd_set rfds, wfds;

		FD_ZERO(&rfds);
		FD_SET(net_fd, &rfds);
		FD_SET(fd_of_shell(tty_state), &rfds);

		FD_ZERO(&wfds);
		if (n_net_buf)
			FD_SET(net_fd, &wfds);

		if (select(FD_SETSIZE, &rfds, &wfds, NULL, NULL) < 0 &&
			errno != EINTR)
		{
			fprintf(stderr, Select_Err);
			exit(1);
		}

		if (FD_ISSET(net_fd, &rfds))
		{
			int n;
			char data[512];

			if ((n = read(net_fd, data, sizeof data)) < 1)
				break;
			append_to_buffer(&buf, &n_buf, data, n);
			do
			{
				n = n_buf;
				parse_buf(&buf, &n_buf);
			}
			while (n != n_buf);
		}

		if (FD_ISSET(fd_of_shell(tty_state), &rfds))
		{
			unsigned char data[512];
			int i, n;

			if ((n = read(fd_of_shell(tty_state),
				data, sizeof data)) < 1)
				break;
			for (i = 0; i < n; i++)
				write_char(term_state, (int)data[i]);
		}

		if (n_net_buf && FD_ISSET(net_fd, &wfds))
		{
			int n;

			if ((n = write(net_fd, net_buf, n_net_buf)) < 1)
				break;
			remove_from_buffer(&net_buf, &n_net_buf, n);
		}
	}

	close(net_fd);
	return 0;
}
