/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <assert.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <X11/Xlib.h>

#include "keys.h"
#include "keymap.h"
#include "attr.h"
#include "redraw.h"
#include "font.h"
#include "delay.h"
#include "proto.h"
#include "auth.h"
#include "display.h"
#include "netcmd.h"
#include "ioqueue.h"
#include "genargv.h"
#include "memory.h"
#include "screens.h"
#include "strings.h"
#include "compose.h"
#include "priv.h"

#ifndef lint
static char sccsid[] = "@(#)jws.c	4.35 9/17/00";
#endif


extern int errno;

static void *font_handle;


#define DEFAULT_WIDTH 80

/* 500 ms before output is expected to be "seen" */
#define USEC_EPS (500 * 1000)


struct net_state
{
	int is_authenticated;
	struct display_auth auth_data;
	int fd, width, height;
	unsigned **off_screen;
	int n_out, n_in;
	char *out, *in;
	int xcursor, ycursor;
};

static struct win_state
{
	int lock;
	struct net_state *net_state;
	void *last_output, *last_bell;
} win_states_add[N_WINDOWS];


static int n_windows_h;
static int cwidth, cheight;
static int focus, focus_page;
static char *path;
static int last_focus = -2;
static int output_to_draw = 1;
static int main_net_fd;

static int save_focus;

static pid_t req_pid = 0;

static int compose_state = 0;
static unsigned char compose_char;


static void clean_exit(void)
{
	char **argv;

	if (req_pid)
		return;

	switch (req_pid = fork())
	{
	case -1:
		fprintf(stderr, Fork_Warn);
		req_pid = 0;
		return;

	case 0:
		break;

	default:
		return;
	}

	argv = gen_argv(Req_Tool_Env, Req_Tool_Default);

	execvp(argv[0], argv);

	fprintf(stderr, Exec_Err, argv[0]);
	exit(1);
}


static void fork_jterm(void)
{
	char **argv;

	switch (fork())
	{
	case -1:
		fprintf(stderr, Fork_Warn);
		return;

	case 0:
		break;

	default:
		return;
	}

	argv = gen_argv(Term_Tool_Env, Term_Tool_Default);

	execvp(argv[0], argv);

	fprintf(stderr, Exec_Err, argv[0]);
	exit(1);
}


static int win_from_net_state(struct net_state *ns)
{
	int i;

	for (i = 0; i < N_WINDOWS; i++)
		if (s_get_active(i) && win_states_add[i].net_state == ns)
			return i;

	return -1;
}


static void try_clear_area(struct net_state *ns,
	int x1, int y1, int x2, int y2)
{
	int k, xc, yc;

	if ((k = win_from_net_state(ns)) < 0 ||
		!s_get_xy_of_visible(k, &xc, &yc))
		return;

	redraw_clear_area(ns->width, xc, yc, x1, y1, x2, y2);
}


static void try_copy_area(struct net_state *ns, int x1, int y1,
	int x2, int y2, int x3, int y3)
{
	int k, xc, yc;

	if ((k = win_from_net_state(ns)) < 0 ||
		!s_get_xy_of_visible(k, &xc, &yc))
		return;

	copy_box(font_handle, x3 + xc, y3 + yc, x1 + xc, y1 + yc,
		x2 + 1 - x1, y2 + 1 - y1);
}


static void bell(void)
{
	font_bell(font_handle);
}


static void draw_number(int attr, int x, int y, int n)
{
	int m;

	for (m = 1; m <= n; m *= 10);
	m /= 10;
	while (m >= 1)
	{
		draw_char(font_handle, x, y, CONV_ATTR('0' + n / m | attr));
		x++;
		n -= n / m * m;
		m /= 10;
	}
}


static void do_redraw_main_win(void)
{
	int i, x;

	clear_box(font_handle, 0, 0, cwidth, cheight, A_Background);

	x = cwidth;
	for (i = N_WINDOWS; i-- > 0;)
		if (s_get_active(i) && !win_states_add[i].lock)
		{
			int attr, state;

			attr = BACKGR;

			if (check_delay(win_states_add[i].last_bell) >= 0)
				attr = CURSOR;

			x--;

			if (i == focus)
				draw_char(font_handle, x, cheight - 1,
					CONV_ATTR('<' | attr));
			else if ((state = check_delay(win_states_add[i].
				last_output)) >= 0)
				draw_char(font_handle, x, cheight - 1,
					CONV_ATTR(Activity_Indicator[state] |
						attr));

			x -= i + 1 > 9 ? 2 : 1;
			draw_number(attr, x, cheight - 1, i + 1);

			x--;
		}

	font_set_size_hint(font_handle, cwidth - x, 1);

	for (i = 0; i < N_WINDOWS; i++)
	{
		int xc, yc;

		if (s_get_xy_of_visible(i, &xc, &yc) &&
			!win_states_add[i].lock)
		{
			draw_number(BACKGR, xc, yc - 1, i + 1);

			if (i == focus && n_windows_h > 1)
			{
				int j;

				xc += i + 1 > 9 ? 2 : 1;
				for (j = 0; Focus_Indicator[j]; j++, xc++)
					draw_char(font_handle, xc, yc - 1,
						CONV_ATTR(Focus_Indicator[j] |
							BACKGR));
			}
		}
	}

	{
		int n;
		char *s;

		s = Jws_Version;
		n = strlen(s);
		x = cwidth - n - 1;
		if (x < 0)
			x = 0;

		while (x < cwidth && n > 0)
		{
			draw_char(font_handle, x, 0,
				(unsigned)*s | A_Background);
			s++;
			x++;
			n--;
		}
	}
}


static void refresh_all_cursors(void)
{
	int i;

	for (i = 0; i < N_WINDOWS; i++)
		if (s_is_visible(i))
			net_refresh_cursor(win_states_add[i].net_state);
}


static void swap_windows(int i, int j)
{
	struct win_state tmp;

	memcpy(&tmp, &win_states_add[i], sizeof tmp);
	memcpy(&win_states_add[i], &win_states_add[j], sizeof tmp);
	memcpy(&win_states_add[j], &tmp, sizeof tmp);

	s_swap(i, j);

	refresh_all_cursors();
}


static void set_focus(int new_focus)
{
	int new_focus_page;
	int was_locked; /* map again without locking */

	if (new_focus == focus || new_focus < -2 || new_focus >= N_WINDOWS ||
		new_focus >= 0 && !s_get_active(new_focus))
		return;

	was_locked = focus >= 0 && win_states_add[focus].lock;

	last_focus = focus;

	if (new_focus >= 0)
		new_focus_page = s_get_page(new_focus);
	else if (new_focus == -2)
		new_focus_page = -1;
	else
		new_focus_page = focus_page;

	if (focus_page >= 0 && new_focus_page != focus_page)
	{
		int i;

		for (i = 0; i < N_WINDOWS; i++)
			if (s_get_active(i) && s_get_page(i) == focus_page)
			{
				start_delay(win_states_add[i].last_bell);
				start_delay(win_states_add[i].last_output);
			}

		if (new_focus_page < 0)
		{
			unmap_font(font_handle);
			was_locked = 0; /* don't map again */
		}
	}

	if (focus >= 0)
	{
		int x, y;

		x = win_states_add[focus].net_state->xcursor;
		y = win_states_add[focus].net_state->ycursor;
		if (x >= 0 && x < win_states_add[focus].net_state->width &&
			y >= 0 && y < win_states_add[focus].net_state->height)
		{
			win_states_add[focus].net_state->off_screen[y][x] &=
				~A_Cursor;
			win_states_add[focus].net_state->off_screen[y][x] |=
				A_Small_Cursor;
		}

		output_to_draw = 1;
	}

	focus = new_focus;

	if (focus >= 0)
	{
		int x, y;

		x = win_states_add[focus].net_state->xcursor;
		y = win_states_add[focus].net_state->ycursor;
		if (x >= 0 && x < win_states_add[focus].net_state->width &&
			y >= 0 && y < win_states_add[focus].net_state->height)
		{
			win_states_add[focus].net_state->off_screen[y][x] &=
				~A_Small_Cursor;
			win_states_add[focus].net_state->off_screen[y][x] |=
				A_Cursor;
		}
	}

	if (new_focus_page != focus_page || was_locked)
	{
		int i;

		if (focus_page < 0 || was_locked ||
			focus >= 0 && win_states_add[focus].lock)
			if (!map_font(font_handle, win_states_add[focus].lock))
			{
				new_focus_page = -1;
				focus = -2;
			}

		focus_page = new_focus_page;

		for (i = 0; i < N_WINDOWS; i++)
		{
			s_set_focus_page(i, focus_page);

			if (s_is_visible(i))
			{
				stop_delay(win_states_add[i].last_bell);
				stop_delay(win_states_add[i].last_output);
			}
		}
	}

	refresh_all_cursors();
}


static void close_window(int k)
{
	int y;

	if (focus == k)
		if (win_states_add[k].lock)
		{
			set_focus(save_focus);
			font_restore_screen(font_handle);
		}
		else
			set_focus(-1);

	free_delay_state(win_states_add[k].last_output);
	free_delay_state(win_states_add[k].last_bell);

	if (s_is_visible(k))
		output_to_draw = 1;

	if (win_states_add[k].net_state->auth_data.length)
		mem_free(win_states_add[k].net_state->auth_data.cookie);

	deregister_io_descriptor(win_states_add[k].net_state->fd);
	close(win_states_add[k].net_state->fd);

	for (y = 0; y < win_states_add[k].net_state->height; y++)
		mem_free(win_states_add[k].net_state->off_screen[y]);
	mem_free(win_states_add[k].net_state->off_screen);
	if (win_states_add[k].net_state->n_out)
		mem_free(win_states_add[k].net_state->out);
	if (win_states_add[k].net_state->n_in)
		mem_free(win_states_add[k].net_state->in);
	mem_free(win_states_add[k].net_state);

	s_set_active(k, 0);
}


static void collect_zombies(void)
{
	pid_t pid;
	int status;

	while ((pid = waitpid(-1, &status, WNOHANG)) > 0)
		if (pid == req_pid)
			if (!WIFEXITED(status) || WEXITSTATUS(status) != 0)
				exit(0);
			else
				req_pid = 0;
}


static void restart(void)
{
	int i;
	char *argv[N_WINDOWS + 3], args[N_WINDOWS][20], n_win_opt[20];

	sprintf(n_win_opt, "-%d", n_windows_h);

	argv[0] = path;
	argv[1] = n_win_opt;
	argv[N_WINDOWS + 2] = NULL;

	for (i = 0; i < N_WINDOWS; i++)
		if (!s_get_active(i))
			argv[i + 2] = "";
		else
		{
			int net_fd;

			net_fd = win_states_add[i].net_state->fd;

			if (fcntl(net_fd, F_SETFD, !FD_CLOEXEC))
			{
				close(net_fd);
				argv[i + 2] = "";
				continue;
			}

			sprintf(args[i], "%d", net_fd);
			argv[i + 2] = args[i];
		}

	exit_font(font_handle);

	execvp(path, argv);

	perror(Restart_Err);
	exit(1);
}


static void start_xlock(void)
{
	char **argv;

	switch (fork())
	{
	case -1:
		fprintf(stderr, Fork_Warn);
		return;

	case 0:
		break;

	default:
		return;
	}

	argv = gen_argv(Lock_Tool_Env, Lock_Tool_Default);

	execvp(argv[0], argv);

	fprintf(stderr, Exec_Err, argv[0]);
	exit(1);
}


static int parse_request(struct net_state *ns, char **buf_in,
	int *size_in, char **buf_out, int *size_out)
{
	int type;
	int rq_len;
	net_cmdT nc;

	if ((type = get_type_from_buffer((void **)buf_in, size_in)) < 0)
		return 0;

	rq_len = get_len_from_buffer((void **)buf_in, size_in);
	nc = NULL;

	if (type == Proto_Dreq_Bell && rq_len == 0)
		nc = make_net_cmd(NC_bell, 0, 0, 0, 0, 0, 0, 0);
	else if (type == Proto_Dreq_Text && rq_len == 2)
	{
		int x, y, v;

		get_pair_from_buffer((void **)buf_in, size_in, &x, &y, 0);
		v = get_word_from_buffer((void **)buf_in, size_in, 1);
		nc = make_net_cmd(NC_text, x, y, 0, 0, 0, 0, v);
	}
	else if (type == Proto_Dreq_Cursor && rq_len == 1)
	{
		int x, y;

		get_pair_from_buffer((void **)buf_in, size_in, &x, &y, 0);
		nc = make_net_cmd(NC_set_cursor, x, y, 0, 0, 0, 0, 0);
	}
	else if (type == Proto_Dreq_Clear_Area && rq_len == 2)
	{
		int x1, y1, x2, y2;

		get_pair_from_buffer((void **)buf_in, size_in, &x1, &y1, 0);
		get_pair_from_buffer((void **)buf_in, size_in, &x2, &y2, 1);
		nc = make_net_cmd(NC_clear, x1, y1, x2, y2, 0, 0, 0);
	}
	else if (type == Proto_Dreq_Copy_Area && rq_len == 3)
	{
		int x1, y1, x2, y2, x3, y3;

		get_pair_from_buffer((void **)buf_in, size_in, &x1, &y1, 0);
		get_pair_from_buffer((void **)buf_in, size_in, &x2, &y2, 1);
		get_pair_from_buffer((void **)buf_in, size_in, &x3, &y3, 2);
		nc = make_net_cmd(NC_copy, x1, y1, x2, y2, x3, y3, 0);
	}
	else if (type == Proto_Ireq_Size && rq_len == 1)
	{
		int w, h;

		get_pair_from_buffer((void **)buf_in, size_in, &w, &h, 0);
		nc = make_net_cmd(NC_resize, w, h, 0, 0, 0, 0, 0);
	}
	else if (type == Proto_Ireq_Lock && rq_len == 0)
		nc = make_net_cmd(NC_lock, 0, 0, 0, 0, 0, 0, 0);
	else if (type == Proto_Ireq_Saver && rq_len == 1)
	{
		int v;

		v = get_word_from_buffer((void **)buf_in, size_in, 0);
		nc = make_net_cmd(NC_saver, 0, 0, 0, 0, 0, 0, v);
	}
	else if (type == Proto_Ireq_Authenticate)
	{
		int ok, i;
		int a, b;

		ok = ns->auth_data.length > 0 &&
			rq_len == (ns->auth_data.length + 1) / 2;

		for (i = 0; ok && i < ns->auth_data.length / 2; i++)
		{
			get_pair_from_buffer((void **)buf_in, size_in,
				&a, &b, i);
			ok &= (char)a == ns->auth_data.cookie[2 * i] &&
				(char)b == ns->auth_data.cookie[2 * i + 1];
		}

		if (ok && i < rq_len)
		{
			get_pair_from_buffer((void **)buf_in, size_in,
				&a, &b, i);
			ok &= (char)a == ns->auth_data.cookie[2 * i];
		}

		nc = make_net_cmd(NC_auth, 0, 0, 0, 0, 0, 0, ok);
	}

	if (nc != NULL)
		exec_and_release_net_cmd(ns, nc);

	return 1;
}


static void parse_requests(struct net_state *ns, char **buf_in,
	int *size_in, char **buf_out, int *size_out)
{
	int skipped;

	skipped = 0;
	while (parse_request(ns, buf_in, size_in, buf_out, size_out))
		remove_block_from_buffer_delay((void **)buf_in,
			size_in, &skipped);

	update_input_buffer((void **)buf_in, size_in, skipped);
}


static void check_write(struct net_state *ns)
{
	set_write_enable(ns->fd, ns->n_out > 0);
}


static int window_by_net_state(struct net_state *ns)
{
	int i;

	for (i = 0; i < N_WINDOWS; i++)
		if (s_get_active(i) && win_states_add[i].net_state == ns)
			return i;

	abort();
}


static void read_from_net(int fd, void *handle)
{
	struct net_state *ns;
	unsigned char buffer[4096];
	int n;

	ns = handle;

	if ((n = read(ns->fd, buffer, sizeof buffer)) < 1)
	{
		if (n < 0 && (errno == EAGAIN || errno == EWOULDBLOCK))
			return;

		close_window(window_by_net_state(ns));
		return;
	}

	append_to_buffer((char **)&ns->in, &ns->n_in, (char *)buffer, n);

	parse_requests(ns, &ns->in, &ns->n_in, &ns->out, &ns->n_out);
	check_write(ns);

	update_delay(win_states_add[window_by_net_state(ns)].last_output);
}


static void write_buffer(int fd, void *handle)
{
	struct net_state *ns;
	int n, m;
	char *out;

	ns = handle;

	n = ns->n_out;
	out = ns->out;

	assert(n >= 0);
	if ((m = write(ns->fd, out, n)) < 1)
	{
		if (m < 0 && (errno == EAGAIN || errno == EWOULDBLOCK))
			return;

		close_window(window_by_net_state(ns));
		return;
	}

	remove_from_buffer(&ns->out, &ns->n_out, m);
	check_write(ns);
}


static void open_net_shell(int k, int fd, struct display_auth *auth_p)
{
	struct win_state *wst;
	struct net_state *nst;
	int x, y;

	wst = &win_states_add[k];

	s_set_active(k, 1);
	s_init_win(k, n_windows_h, DEFAULT_WIDTH);
	s_set_focus_page(k, focus_page);

	nst = wst->net_state = mem_alloc(sizeof *wst->net_state);
	nst->width = DEFAULT_WIDTH;
	nst->height = cheight - 2;

	nst->off_screen = mem_alloc((cheight - 2) * sizeof *nst->off_screen);
	for (y = 0; y < cheight - 2; y++)
	{
		nst->off_screen[y] = mem_alloc(DEFAULT_WIDTH *
			sizeof **nst->off_screen);
		for (x = 0; x < DEFAULT_WIDTH; x++)
			nst->off_screen[y][x] = ' ';
	}

	nst->is_authenticated = 0;
	nst->auth_data.length = 0;
	if (auth_p != NULL)
		memcpy(&nst->auth_data, auth_p, sizeof *auth_p);

	wst->lock = 0;
	nst->fd = fd;
	nst->n_out = 0;
	nst->n_in = 0;
	nst->xcursor = -1;
	nst->ycursor = -1;

	register_io_descriptor(fd);
	set_read_handler(fd, read_from_net, nst);
	set_read_enable(fd, 1);
	set_write_handler(fd, write_buffer, nst);

	if (s_is_visible(k))
	{
		output_to_draw = 1;
		refresh_all_cursors();
	}

	wst->last_output = init_delay_state(USEC_EPS,
		strlen(Activity_Indicator) - 1);
	wst->last_bell = init_delay_state(USEC_EPS, 0);

	append_header_to_buffer((void **)&nst->out, &nst->n_out,
		Proto_Ev_Size, 1);
	append_pair_to_buffer((void **)&nst->out, &nst->n_out, DEFAULT_WIDTH,
		cheight - 2);

	check_write(nst);
}


/* ----- */


void net_clear_area(struct net_state *ns,
	int x1, int y1, int x2, int y2)
{
	int x, y;

	if (x2 >= ns->width)
		x2 = ns->width - 1;
	if (y2 >= ns->height)
		y2 = ns->height - 1;

	if (x1 > x2 || y1 > y2)
		return;

	for (y = y1; y <= y2; y++)
		for (x = x1; x <= x2; x++)
			ns->off_screen[y][x] = ' ';

	try_clear_area(ns, x1, y1, x2, y2);

	output_to_draw = 1;
}


void net_copy_area(struct net_state *ns, int x1, int y1,
	int x2, int y2, int xto, int yto)
{
	int x, y, dx, dy, x3, y3;

	dx = xto - x1;
	dy = yto - y1;

	if (!dx && !dy)
		return;

	if (x2 >= ns->width - dx)
		x2 = ns->width - 1 - dx;
	if (y2 >= ns->height - dy)
		y2 = ns->height - 1 - dy;

	if (x1 > x2 || y1 > y2)
		return;

	x3 = x2;
	y3 = y2;

	if (x2 >= ns->width)
		x2 = ns->width - 1;
	if (y2 >= ns->height)
		y2 = ns->height - 1;

	if (dy < 0 || dy == 0 && dx < 0)
		for (y = y1; y <= y2; y++)
			for (x = x1; x <= x2; x++)
				ns->off_screen[y + dy][x + dx] =
					ns->off_screen[y][x];
	else
		for (y = y2; y >= y1; y--)
			for (x = x2; x >= x1; x--)
				ns->off_screen[y + dy][x + dx] =
					ns->off_screen[y][x];

	try_copy_area(ns, x1, y1, x2, y2, x1 + dx, y1 + dy);

	if (y3 > y2)
		net_clear_area(ns, x1 + dx, y2 + 1 + dy, x2 + dx, y3 + dy),
		y3 = y2;

	if (x3 > x2)
		net_clear_area(ns, x2 + 1 + dx, y1 + dy, x3 + dx, y2 + dy);

	output_to_draw = 1;
}


void net_write_char(struct net_state *ns, int x, int y, int ch)
{
	if (x >= ns->width || y >= ns->height)
		return;

	ns->off_screen[y][x] = CONV_ATTR(ch);

	output_to_draw = 1;
}


void net_hide_cursor(struct net_state *ns)
{
	int x, y;

	x = ns->xcursor;
	y = ns->ycursor;
	if (x >= 0 && x < ns->width && y >= 0 && y < ns->height)
	{
		ns->off_screen[y][x] &= ~(A_Cursor | A_Small_Cursor);
		output_to_draw = 1;
	}
}


void net_refresh_cursor(struct net_state *ns)
{
	int x, y;

	x = ns->xcursor;
	y = ns->ycursor;
	if (x >= 0 && x < ns->width && y >= 0 && y < ns->height)
	{
		ns->off_screen[y][x] |= (focus >= 0 &&
			win_states_add[focus].net_state == ns ?
			A_Cursor : A_Small_Cursor);
		output_to_draw = 1;
	}
}


void net_set_cursor(struct net_state *ns, int x, int y)
{
	int xc, yc;

	xc = ns->xcursor;
	yc = ns->ycursor;
	if (xc >= 0 && xc < ns->width && yc >= 0 && yc < ns->height)
		ns->off_screen[yc][xc] &= ~(A_Cursor | A_Small_Cursor);
	ns->xcursor = x;
	ns->ycursor = y;
	net_refresh_cursor(ns);
}


void net_resize(struct net_state *ns, int w, int h)
{
	int i, y;

	if (w < 1 || w > ns->width || h < 1 || h > ns->height)
		return;

	if ((i = win_from_net_state(ns)) < 0)
		return;

	for (y = h; y < ns->height; y++)
		mem_free(ns->off_screen[y]);
	ns->height = h;

	for (y = 0; y < h; y++)
		ns->off_screen[y] = mem_realloc(ns->off_screen[y],
			w * sizeof **ns->off_screen);
	ns->width = w;

	if (win_states_add[i].lock)
		s_move(i, (cwidth - w) / 2, (cheight - 2 - h) * 2 / 5 + 1);

	output_to_draw = 1;

	append_header_to_buffer((void **)&ns->out, &ns->n_out,
		Proto_Ev_Size, 1);
	append_pair_to_buffer((void **)&ns->out, &ns->n_out, w, h);

	check_write(ns);
}


void net_lock_window(struct net_state *ns)
{
	int i;

	i = win_from_net_state(ns);

	if (i >= 0 && (focus < 0 || !win_states_add[focus].lock))
	{
		append_header_to_buffer((void **)&ns->out, &ns->n_out,
			Proto_Ev_Lock, 0);

		save_focus = focus == i ? -1 : focus;
		set_focus(-1);

		win_states_add[i].lock = 1;

		s_move_page(i, N_WINDOWS);
		s_set_focus_page(i, focus_page);
		s_move(i, (cwidth - ns->width) / 2,
			(cheight - 2 - ns->height) * 2 / 5 + 1);

		set_focus(i);
		refresh_all_cursors();
	}
	else
		append_header_to_buffer((void **)&ns->out, &ns->n_out,
			Proto_Ev_Nolock, 0);

	check_write(ns);
}


void net_set_saver(struct net_state *ns, int timeout)
{
	if (focus < 0 || !win_states_add[focus].lock ||
		win_states_add[focus].net_state != ns)
		return;

	font_save_screen(font_handle, timeout);
}


void net_bell(struct net_state *ns)
{
	int i;

	if ((i = win_from_net_state(ns)) < 0)
		return;

	update_delay(win_states_add[i].last_bell);

	bell();
}


int *net_auth_flag(struct net_state *ns)
{
	return &ns->is_authenticated;
}


/* ----- */


static void do_input(int key, char *str, int n, int on_root)
{
	int i;
	int prev_compose_state, is_input;

	/* reset compose mode per default */
	prev_compose_state = compose_state;
	compose_state = 0;

	is_input = key == Action_Input || key == Action_Compose_Input;

	if (!is_input && focus >= 0 && win_states_add[focus].lock)
		return;

	if (focus < 0 || !win_states_add[focus].lock)
		for (i = 0; i < N_WINDOWS; i++)
			if (s_get_active(i) && win_states_add[i].lock)
			{
				/* don't retry on every keypress,
				   only on other actions */
				if (!is_input)
					set_focus(i);

				return;
			}

	switch (key)
	{
	case Action_Lock:
		start_xlock();
		return;

	case Action_X11:
		set_focus(-2);
		return;

	case Action_Restart:
		restart();
		return;

	case Action_Exit:
		clean_exit();
		return;

	case Action_Newterm:
		fork_jterm();
		return;

	case Action_Input:
		break;

	case Action_Compose_Input:
		compose_state = 1;
		return;

	case Action_Win_Goto:
		if (n >= 0 && n < N_WINDOWS)
		{
			int new_focus;

			new_focus = n;
			if (s_get_active(new_focus))
			{
				set_focus(new_focus);
				return;
			}
		}
		bell();
		return;

	case Action_Win_Move:
		if (n >= 0 && n < N_WINDOWS && focus >= 0)
		{
			int new, old;
			int move;

			new = n;
			old = focus;
			if (old == new)
				return;

			set_focus(-1);

			move = !s_get_active(new);

			if (move)
			{
				s_set_active(new, 1);
				s_init_win(new, n_windows_h, DEFAULT_WIDTH);
				/* don't set the focus page for new,
				   it must not become visible */
			}

			swap_windows(new, old);

			if (move)
				s_set_active(old, 0);

			set_focus(new);
			return;
		}
		bell();
		return;

	default:
		/* no action, restore compose state */
		compose_state = prev_compose_state;
		return;
	}

	if (on_root || focus < 0)
		return;

	if (prev_compose_state && n != 1)
		bell();
	else if (prev_compose_state && (str[0] & 0x7f) >= 0x20)
	{
		int i;

		if (prev_compose_state == 1)
		{
			compose_char = str[0];
			compose_state = 2;
			return;
		}

		n = 0;
		for (i = 0; jws_compose_table[i][0]; i++)
			if (jws_compose_table[i][0] == compose_char &&
				jws_compose_table[i][1] ==
					(unsigned char)str[0])
			{
				compose_char = jws_compose_table[i][2];
				str = (char *)&compose_char;
				n = 1;
				break;
			}

		if (!n)
		{
			bell();
			return;
		}
	}

	for (i = 0; i < n; i++)
	{
		if (!str[i])
			continue;

		append_header_to_buffer((void **)&win_states_add[focus].net_state->out,
			&win_states_add[focus].net_state->n_out,
			Proto_Ev_Key, 1);
		append_word_to_buffer((void **)&win_states_add[focus].net_state->out,
			&win_states_add[focus].net_state->n_out,
			(int)(unsigned char)str[i]);
	}

	check_write(win_states_add[focus].net_state);
}


static void accept_a_client(int master_fd, void *handle)
{
	int i, net_fd, *wait_flagP;
	struct display_auth auth;

	for (i = 0; i < N_WINDOWS && s_get_active(i); i++);

	net_fd = accept_display_client(master_fd, &auth);

	if (net_fd < 0)
		return;

	if (i >= N_WINDOWS)
	{
		if (auth.length)
			mem_free(auth.cookie);
		close(net_fd);
		return;
	}

	open_net_shell(i, net_fd, &auth);

	wait_flagP = handle;
	if (*wait_flagP)
	{
		*wait_flagP = 0;
		set_focus(0);
	}
}


/* ----- */


int main(int argc, char **argv)
{
	int i, wait_for_client;
	int skipped_args;

	init_and_lower_priv();

	path = argv[0];

	{
		struct sigaction sa;

		sa.sa_handler = (void (*)())collect_zombies;
		sigemptyset(&sa.sa_mask);
		sa.sa_flags = 0;

		if (sigaction(SIGCHLD, &sa, NULL))
		{
			fprintf(stderr, SIGCHLD_Handler_Err);
			exit(1);
		}
	}

	if (argc > 1 && argv[1][0] == '-')
	{
		n_windows_h = atoi(argv[1] + 1);
		skipped_args = 1;
	}
	else
	{
		n_windows_h = 2;
		skipped_args = 0;
	}

	assert(n_windows_h > 0);

	collect_zombies();

	cwidth = n_windows_h * DEFAULT_WIDTH;
	font_handle = init_font(do_input, &cwidth, &cheight);

	assert(n_windows_h * DEFAULT_WIDTH <= cwidth);
	assert(cheight >= 5);

	init_redraw(font_handle, cwidth, cheight);

	focus = -2;
	focus_page = -1;

	wait_for_client = 1;
	for (i = 0; i < N_WINDOWS && i + 1 + skipped_args < argc; i++)
		if (*argv[i + 1 + skipped_args])
		{
			int net_fd;

			net_fd = atoi(argv[i + 1 + skipped_args]);

			if (fcntl(net_fd, F_SETFD, FD_CLOEXEC))
			{
				close(net_fd);
				continue;
			}

			open_net_shell(i, net_fd, NULL),
			wait_for_client = 0;
		}

	for (i = 0; i < N_WINDOWS && focus < 0; i++)
		set_focus(i);

	if ((main_net_fd = bind_to_display(default_display_name())) < 0)
	{
		fprintf(stderr, Bind_Socket_Err);
		exit(1);
	}

	if (wait_for_client)
		fork_jterm();

	register_io_descriptor(main_net_fd);
	set_read_handler(main_net_fd, accept_a_client, &wait_for_client);
	set_read_enable(main_net_fd, 1);

	for (;;)
	{
		for (i = 0; i < N_WINDOWS && !s_get_active(i); i++);
		if (!wait_for_client && i >= N_WINDOWS)
			break;

		if (output_to_draw)
		{
			do_redraw_main_win();
			for (i = 0; i < N_WINDOWS; i++)
			{
				int xc, yc;

				if (s_get_xy_of_visible(i, &xc, &yc))
					redraw_copy_to_screen(win_states_add[i].net_state->off_screen,
						xc, yc,
						win_states_add[i].net_state->width,
						win_states_add[i].net_state->height);
			}

			output_to_draw = 0;
		}

		do_select();
	}

	return 0;
}
