/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "memory.h"
#include "strings.h"

#ifndef lint
static char sccsid[] = "@(#)memory.c	4.1 2/27/99";
#endif


#ifdef DEBUG_MEMORY

#ifndef D_M_SIZE
#define D_M_SIZE 8
#endif

#ifndef D_M_HEADER_SIZE
#define D_M_HEADER_SIZE D_M_SIZE
#endif

#ifndef D_M_TRAILER_SIZE
#define D_M_TRAILER_SIZE D_M_SIZE
#endif

#else

#undef D_M_HEADER_SIZE
#undef D_M_TRAILER_SIZE

#define D_M_HEADER_SIZE 0
#define D_M_TRAILER_SIZE 0

#endif


static void memory_error(void)
{
	fprintf(stderr, Out_Of_Memory_Err);
	exit(1);
}


#ifdef DEBUG_MEMORY
static void set_fence(unsigned char *p, int size)
{
	int i;

	memcpy(p, &size, sizeof(int));
	for (i = sizeof(int); i < D_M_HEADER_SIZE; i++)
		p[i] = 0x5a + i;
	for (i = 0; i < D_M_TRAILER_SIZE; i++)
		p[D_M_HEADER_SIZE + size + i] = 0x5a + i;
}


static void check_fence(unsigned char *p)
{
	int i, size;

	memcpy(&size, p, sizeof(int));
	assert(size > 0);
	for (i = sizeof(int); i < D_M_HEADER_SIZE; i++)
		assert(p[i] == (unsigned char)(0x5a + i));
	for (i = 0; i < D_M_TRAILER_SIZE; i++)
		assert(p[D_M_HEADER_SIZE + size + i] ==
			(unsigned char)(0x5a + i));
}


static void clear_fence(unsigned char *p)
{
	int i, size;

	memcpy(&size, p, sizeof(int));
	for (i = 0; i < D_M_HEADER_SIZE; i++)
		p[i] = i;
	for (i = 0; i < D_M_TRAILER_SIZE; i++)
		p[D_M_HEADER_SIZE + size + i] = i;
}
#endif


/*
**	internal and external
*/


void *mem_alloc(int size)
{
	unsigned char *m;

	assert(size > 0);

	if ((m = (void *)malloc(size + D_M_HEADER_SIZE +
		D_M_TRAILER_SIZE)) == NULL)
		memory_error();

#ifdef DEBUG_MEMORY
	set_fence(m, size);
#endif

	return m + D_M_HEADER_SIZE;
}


void *mem_realloc(void *m, int size)
{
	unsigned char *p;

	assert(size > 0);

	p = m;
	p -= D_M_HEADER_SIZE;

#ifdef DEBUG_MEMORY
	check_fence(p);
	clear_fence(p);
#endif

	if ((p = (void *)realloc(p, size + D_M_HEADER_SIZE +
		D_M_TRAILER_SIZE)) == NULL)
		memory_error();

#ifdef DEBUG_MEMORY
	set_fence(p, size);
#endif

	return p + D_M_HEADER_SIZE;
}


void mem_free(void *m)
{
	unsigned char *p;

	p = m;
	p -= D_M_HEADER_SIZE;

#ifdef DEBUG_MEMORY
	check_fence(p);
	clear_fence(p);
#endif

	free(p);
}


void mem_verify(void *m)
{
#ifdef DEBUG_MEMORY
	unsigned char *p;

	p = m;
	p -= D_M_HEADER_SIZE;

	check_fence(p);
#endif
}


/*
**	external only
*/


void append_to_buffer(char **buf, int *size, char *data, int n)
{
	int m;

	assert(n > 0);
	assert(*size >= 0);

	if (m = *size)
		*buf = mem_realloc(*buf, m + n);
	else
		*buf = mem_alloc(n);

	*size += n;
	memcpy(*buf + m, data, n);
}


void remove_from_buffer(char **buf, int *size, int n)
{
	assert(n > 0);
	assert(n <= *size);

	if (!(*size -= n))
		mem_free(*buf);
	else
		memmove(*buf, &(*buf)[n], *size);
}
