/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/keysym.h>

#include "keymap.h"
#include "memory.h"
#include "strings.h"

#ifndef lint
static char sccsid[] = "@(#)parsekmap.c	4.2 9/23/00";
#endif


struct modifier
{ 
	char name;
	unsigned mask;
};

struct action_parse
{
	char *name;
	int type; /* Action_*, but not Action_Input */
	int n; /* index for Action_Win_* */
};


#define ALT_MOD		0
#define META_MOD	1

static struct modifier modifiers[] =
{
	{ 'a', 0 },
	{ 'm', 0 },
	{ 's', ShiftMask },
	{ 'l', LockMask },
	{ 'c', ControlMask },
	{ '1', Mod1Mask },
	{ '2', Mod2Mask },
	{ '3', Mod3Mask },
	{ '4', Mod4Mask },
	{ '5', Mod5Mask }
};

static struct action_parse actions[] =
{
	{ "w01", Action_Win_Goto, 0 }, { "w02", Action_Win_Goto, 1 },
	{ "w03", Action_Win_Goto, 2 }, { "w04", Action_Win_Goto, 3 },
	{ "w05", Action_Win_Goto, 4 }, { "w06", Action_Win_Goto, 5 },
	{ "w07", Action_Win_Goto, 6 }, { "w08", Action_Win_Goto, 7 },
	{ "w09", Action_Win_Goto, 8 }, { "w10", Action_Win_Goto, 9 },
	{ "w11", Action_Win_Goto, 10 }, { "w12", Action_Win_Goto, 11 },
	{ "w13", Action_Win_Goto, 12 }, { "w14", Action_Win_Goto, 13 },
	{ "w15", Action_Win_Goto, 14 }, { "w16", Action_Win_Goto, 15 },
	{ "w17", Action_Win_Goto, 16 }, { "w18", Action_Win_Goto, 17 },
	{ "w19", Action_Win_Goto, 18 }, { "w20", Action_Win_Goto, 19 },
	{ "w21", Action_Win_Goto, 20 }, { "w22", Action_Win_Goto, 21 },
	{ "w23", Action_Win_Goto, 22 }, { "w24", Action_Win_Goto, 23 },
	{ "w25", Action_Win_Goto, 24 }, { "w26", Action_Win_Goto, 25 },
	{ "w27", Action_Win_Goto, 26 }, { "w28", Action_Win_Goto, 27 },
	{ "w29", Action_Win_Goto, 28 }, { "w30", Action_Win_Goto, 29 },
	{ "w31", Action_Win_Goto, 30 }, { "w32", Action_Win_Goto, 31 },
	{ "s01", Action_Win_Move, 0 }, { "s02", Action_Win_Move, 1 },
	{ "s03", Action_Win_Move, 2 }, { "s04", Action_Win_Move, 3 },
	{ "s05", Action_Win_Move, 4 }, { "s06", Action_Win_Move, 5 },
	{ "s07", Action_Win_Move, 6 }, { "s08", Action_Win_Move, 7 },
	{ "s09", Action_Win_Move, 8 }, { "s10", Action_Win_Move, 9 },
	{ "s11", Action_Win_Move, 10 }, { "s12", Action_Win_Move, 11 },
	{ "s13", Action_Win_Move, 12 }, { "s14", Action_Win_Move, 13 },
	{ "s15", Action_Win_Move, 14 }, { "s16", Action_Win_Move, 15 },
	{ "s17", Action_Win_Move, 16 }, { "s18", Action_Win_Move, 17 },
	{ "s19", Action_Win_Move, 18 }, { "s20", Action_Win_Move, 19 },
	{ "s21", Action_Win_Move, 20 }, { "s22", Action_Win_Move, 21 },
	{ "s23", Action_Win_Move, 22 }, { "s24", Action_Win_Move, 23 },
	{ "s25", Action_Win_Move, 24 }, { "s26", Action_Win_Move, 25 },
	{ "s27", Action_Win_Move, 26 }, { "s28", Action_Win_Move, 27 },
	{ "s29", Action_Win_Move, 28 }, { "s30", Action_Win_Move, 29 },
	{ "s31", Action_Win_Move, 30 }, { "s32", Action_Win_Move, 31 },
	{ "lock", Action_Lock }, { "x11", Action_X11 },
	{ "restart", Action_Restart }, { "exit", Action_Exit },
	{ "launch", Action_Newterm }, { "compose", Action_Compose_Input }
};


static void find_modifier(Display *display, XModifierKeymap *mods,
	KeySym sym, unsigned *mask_p)
{
	KeyCode code;
	int i, j;

	code = XKeysymToKeycode(display, sym);
	if (code == 0)
		return;

	for (i = 0; i < 8; i++)
		for (j = 0; j < mods->max_keypermod; j++)
			if (mods->modifiermap[j + mods->max_keypermod * i] ==
				code)
			{
				*mask_p = 1 << i;
				return;
			}
}


static void get_modifiers(Display *display)
{
	XModifierKeymap *mods;

	mods = XGetModifierMapping(display);

	find_modifier(display, mods, XK_Alt_L, &modifiers[ALT_MOD].mask);
	find_modifier(display, mods, XK_Alt_R, &modifiers[ALT_MOD].mask);
	find_modifier(display, mods, XK_Meta_L, &modifiers[META_MOD].mask);
	find_modifier(display, mods, XK_Meta_R, &modifiers[META_MOD].mask);

	XFreeModifiermap(mods);
}


static int conv_to_upper(int ch)
{
	if (ch >= 'a' && ch <= 'z')
		return ch - 'a' + 'A';
	else
		return ch;
}


static int parse_hex(char *str)
{
	int first, second;

	first = conv_to_upper((int)str[0]);
	if (first >= '0' && first <= '9')
		first -= '0';
	else if (first >= 'A' && first <= 'F')
		first -= 'A' - 10;
	else
		return -1;

	second = conv_to_upper((int)str[1]);
	if (second >= '0' && second <= '9')
		second -= '0';
	else if (second >= 'A' && second <= 'F')
		second -= 'A' - 10;
	else
		return -1;

	return first * 16 + second;
}


static int parse_action(char **str_p, int *type_p, int *arg_p)
{
	if (**str_p >= '!' && **str_p <= '~' && **str_p != '\\')
	{
		*type_p = Action_Input;
		*arg_p = (unsigned char)**str_p;
		(*str_p)++;
		return 1;
	}

	if (**str_p != '\\')
		return 0;

	(*str_p)++;
	if (**str_p == '\\')
	{
		(*str_p)++;
		*type_p = Action_Input;
		*arg_p = '\\';
		return 1;
	}

	if (**str_p == '^')
	{
		char name;

		(*str_p)++;
		name = conv_to_upper((int)**str_p);
		(*str_p)++;

		if (name < 'A' || name > '_')
			return 0;

		*type_p = Action_Input;
		*arg_p = name - 'A' + 1;
		return 1;
	}

	if (**str_p == '!')
	{
		int i;

		(*str_p)++;
		for (i = 0; i < sizeof actions / sizeof *actions; i++)
			if (!strncmp(*str_p, actions[i].name,
				strlen(actions[i].name)))
			{
				*type_p = actions[i].type;
				*arg_p = actions[i].n;
				*str_p += strlen(actions[i].name);
				return 1;
			}

		return 0;
	}

	{
		int hex;

		if ((hex = parse_hex(*str_p)) < 0)
			return 0;

		*str_p += 2;

		*type_p = Action_Input;
		*arg_p = hex;
		return 1;
	}
}


static int parse_action_list(char **str_p, int *type_p,
	int *n_p, char **data_p)
{
	int type, arg, n;
	char *data;

	if (!parse_action(str_p, &type, &arg))
		return 0;

	if (type != Action_Input)
	{
		if (**str_p == ',')
			return 0;

		*type_p = type;
		*n_p = arg;
		*data_p = NULL;
		return 1;
	}

	data = mem_alloc(1);
	data[0] = arg;

	for (n = 1; **str_p == ','; n++)
	{
		(*str_p)++;

		if (!parse_action(str_p, &type, &arg))
		{
			mem_free(data);
			return 0;
		}

		if (type != Action_Input)
		{
			mem_free(data);
			return 0;
		}

		data = mem_realloc(data, n + 1);
		data[n] = arg;
	}

	*type_p = type;
	*n_p = n;
	*data_p = data;
	return 1;
}


static struct action *parse_mapping(char **str_p)
{
	struct action *a;
	unsigned bits, mask;
	int grab, type, arg;
	char *data;

	for (bits = 0; **str_p; (*str_p)++)
	{
		int i;

		for (i = 0; i < sizeof modifiers / sizeof *modifiers; i++)
			if (**str_p == modifiers[i].name)
			{
				bits |= modifiers[i].mask;
				break;
			}

		if (i >= sizeof modifiers / sizeof *modifiers)
			break;
	}

	grab = 0;
	mask = ShiftMask | LockMask | ControlMask | Mod1Mask |
		Mod2Mask | Mod3Mask | Mod4Mask | Mod5Mask;

	if (**str_p == '?')
		for ((*str_p)++; **str_p; (*str_p)++)
		{
			int i;

			for (i = 0; i < sizeof modifiers / sizeof *modifiers;
				i++)
				if (**str_p == modifiers[i].name)
				{
					mask &= ~modifiers[i].mask;
					break;
				}

			if (i >= sizeof modifiers / sizeof *modifiers)
				break;
		}
	else if (**str_p == '!')
	{
		(*str_p)++;
		grab = 1;
	}

	mask |= bits;

	if (*(*str_p)++ != '=')
		return NULL;

	if (!parse_action_list(str_p, &type, &arg, &data))
		return NULL;

	a = mem_alloc(sizeof *a);

	a->mod_bits = bits;
	a->mod_mask = mask;
	a->grab = grab;
	a->type = type;
	a->n = arg;
	a->data = data;

	return a;
}


static void free_mapping_list(int n, struct action *list)
{
	int i;

	for (i = 0; i < n; i++)
		if (list[i].type == Action_Input)
			mem_free(list[i].data);

	if (n)
		mem_free(list);
}


static int parse_mapping_list(char **str_p, int *n_p, struct action **list_p)
{
	int n;
	struct action *list, *a;

	n = 0;
	do
	{
		if ((a = parse_mapping(str_p)) == NULL)
		{
			free_mapping_list(n, list);
			return 0;
		}

		if (n++)
			list = mem_realloc(list, n * sizeof *list);
		else
			list = mem_alloc(sizeof *list);

		memcpy(&list[n - 1], a, sizeof *list);
		mem_free(a);
	}
	while (*(*str_p)++ == '\t');

	(*str_p)--;

	*n_p = n;
	*list_p = list;
	return 1;
}


static int parse_key_name(Display *display, char *name, KeyCode *key_p)
{
	KeySym sym;

	if (name[0] == '=')
	{
		int hex;

		if (strlen(name) != 3 || (hex = parse_hex(name + 1)) <= 0)
			return 0;
		return (*key_p = (KeyCode)hex) == hex;
	}
	else
	{
		sym = XStringToKeysym(name);
		if (sym == NoSymbol)
			return 0;

		return (*key_p = XKeysymToKeycode(display, sym)) != 0;
	}
}


static int parse_line(Display *display, char *str, KeyCode *key_p,
	int *n_p, struct action **list_p)
{
	char *end, *name;
	int len;

	if ((end = strchr(str, '\t')) == NULL)
		len = strlen(str);
	else
		len = end - str;

	name = mem_alloc(len + 1);
	strncpy(name, str, len);
	name[len] = 0;

	if (!parse_key_name(display, name, key_p))
	{
		mem_free(name);
		return 0;
	}

	mem_free(name);

	if (end == NULL)
	{
		*n_p = 0;
		return 1;
	}

	str = end + 1;
	if (!parse_mapping_list(&str, n_p, list_p))
		return 0;

	if (*str)
	{
		free_mapping_list(*n_p, *list_p);
		return 0;
	}

	return 1;
}


static char *read_line(FILE *f)
{
	char buffer[1024];
	char *result;
	int len;

	result = mem_alloc(1);
	len = 0;

	while (fgets(buffer, sizeof buffer, f) != NULL)
	{
		int n;

		n = strlen(buffer);
		if (!n)
		{
			mem_free(result);
			return NULL;
		}

		if (buffer[n - 1] == '\n')
			n--;

		result = mem_realloc(result, len + n + 1);
		strncpy(result + len, buffer, n);
		len += n;

		if (buffer[n])
		{
			result[len] = 0;
			return result;
		}
	}

	mem_free(result);
	return NULL;
}


/*
**	external
*/


void parse_file(Display *display, char *filename, struct key keys[N_KEYCODES])
{
	FILE *f;
	char *line;
	int i;

	for (i = 0; i < N_KEYCODES; i++)
		keys[i].in_use = keys[i].n = 0;

	get_modifiers(display);

	if ((f = fopen(filename, "r")) == NULL)
	{
		fprintf(stderr, Keymap_Read_Warn, filename);
		return;
	}
	
	for (i = 1; (line = read_line(f)) != NULL; i++)
	{
		KeyCode key;
		int n;
		struct action *list;

		if (!line[0] || line[0] == '\t')
			continue; /* skip empty and indented lines */

		if (!parse_line(display, line, &key, &n, &list))
			fprintf(stderr, Keymap_Parse_Warn, i);
		else if (key < 0 || key >= N_KEYCODES) /* cc may warn here */
			fprintf(stderr, Keymap_Keycode_Warn, i);
		else if (keys[key].in_use)
			fprintf(stderr, Keymap_Redef_Warn, i);
		else
		{
			keys[key].in_use = 1;
			keys[key].n = n;
			keys[key].list = list;
		}

		mem_free(line);
	}

	fclose(f);
}
