/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "proto.h"
#include "auth.h"
#include "display.h"
#include "memory.h"
#include "popup.h"
#include "strings.h"

#ifndef lint
static char sccsid[] = "@(#)popup.c	3.4 2/8/99";
#endif


static void *ibuf, *obuf;
static int n_ibuf = 0, n_obuf = 0;
static int net_fd, skip_keys = 0;


static int get_key(void)
{
	int type, ev_len, ch;

	for (ch = 0; !ch;)
	{
		if ((type = get_type_from_buffer(&ibuf, &n_ibuf)) < 0)
			break;

		ev_len = get_len_from_buffer(&ibuf, &n_ibuf);

		if (type == Proto_Ev_Nolock)
		{
			fprintf(stderr, Lock_Rejected_Err);
			exit(1);
		}

		if (type == Proto_Ev_Key && ev_len > skip_keys)
		{
			ch = get_word_from_buffer(&ibuf, &n_ibuf, skip_keys++);
			continue;
		}

		remove_block_from_buffer(&ibuf, &n_ibuf);
		skip_keys = 0;
	}

	return ch;
}


/*
**	internal and external
*/


void popup_write_char(int *x, int *y, int chr)
{
	append_header_to_buffer(&obuf, &n_obuf, Proto_Dreq_Text, 2);
	append_pair_to_buffer(&obuf, &n_obuf, *x, *y);
	append_word_to_buffer(&obuf, &n_obuf, chr);

	(*x)++;
}


/*
**	external
*/


void popup_write_string(int *x, int *y, char *str)
{
	while (*str)
		popup_write_char(x, y, (int)*str++);
}


void init_popup(int width, int height, int screen_saver)
{
	struct display_auth auth;

	if ((net_fd = connect_to_display(default_display_name(), &auth)) < 0)
	{
		fprintf(stderr, Connect_Err);
		exit(1);
	}

	if (auth.length)
	{
		append_raw_to_buffer(&obuf, &n_obuf, Proto_Ireq_Authenticate,
			auth.length, auth.cookie);
		mem_free(auth.cookie);
	}

	append_header_to_buffer(&obuf, &n_obuf, Proto_Ireq_Size, 1);
	append_pair_to_buffer(&obuf, &n_obuf, width, height);

	append_header_to_buffer(&obuf, &n_obuf, Proto_Ireq_Lock, 0);

	if (screen_saver)
	{
		append_header_to_buffer(&obuf, &n_obuf, Proto_Ireq_Saver, 1);
		append_word_to_buffer(&obuf, &n_obuf, screen_saver);
	}
}


void close_popup(void)
{
}


int popup_get_keys(char *keys, int n_keys)
{
	int n, ch;

	if (n_obuf)
	{
		int m;

		if ((m = write(net_fd, obuf, n_obuf)) < 1)
		{
			fprintf(stderr, Write_To_Server_Err);
			exit(1);
		}
		remove_from_buffer((char **)&obuf, &n_obuf, m);
	}

	n = 0;
	if (!(ch = get_key()))
	{
		char in[512];
		int m;

		if ((m = read(net_fd, in, sizeof in)) < 1)
		{
			fprintf(stderr, Read_From_Server_Err);
			exit(1);
		}
		append_to_buffer((char **)&ibuf, &n_ibuf, (void *)in, m);

		ch = get_key();
	}

	while (ch)
	{
		if (ch > 0 && ch < 255)
			keys[n++] = ch;
		if (n >= n_keys)
			break;
		ch = get_key();
	}

	return n;
}


void popup_bell(void)
{
	append_header_to_buffer(&obuf, &n_obuf, Proto_Dreq_Bell, 0);
}


void popup_set_cursor(int x, int y)
{
	append_header_to_buffer(&obuf, &n_obuf, Proto_Dreq_Cursor, 1);
	append_pair_to_buffer(&obuf, &n_obuf, x, y);
}
