/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <assert.h>

#include "proto.h"
#include "memory.h"

#ifndef lint
static char sccsid[] = "@(#)proto.c	4.1 3/3/99";
#endif


static void get_pair(void **buffer, int *size, int *lohalf,
	int *hihalf, int offset)
{
	unsigned char *data;

	offset *= 2;

	assert(offset >= 0);
	assert(offset + 2 <= *size);

	data = (unsigned char *)*buffer + offset;
	*lohalf = data[0];
	*hihalf = data[1];
}


static int get_word(void **buffer, int *size, int offset)
{
	int lohalf, hihalf;

	get_pair(buffer, size, &lohalf, &hihalf, offset);

	return hihalf * 256 + lohalf;
}


static int get_header(void **buffer, int *size, int *type, int *data_len)
{
	unsigned short *data;
	int rq_type, len;

	data = *buffer;
	if (*size < 2)
		return 0;

	rq_type = get_word(buffer, size, 0);
	data++;

	len = (*data_len = rq_type >> 14) + 1;
	if (len == 4)
	{
		if (*size < 4)
			return 0;

		len = (*data_len = get_word(buffer, size, 1)) + 2;
	}

	if (*size < 2 * len)
		return 0;

	*type = rq_type & (1 << 14) - 1;

	return 2 * len;
}


/*
**	internal and external
*/


void append_pair_to_buffer(void **buffer, int *size, int lohalf, int hihalf)
{
	char data[2];

	assert(lohalf >= 0);
	assert(lohalf < 256);
	assert(hihalf >= 0);
	assert(hihalf < 256);

	data[0] = lohalf;
	data[1] = hihalf;

	append_to_buffer((char **)buffer, size, (void *)data, sizeof data);
}


void append_word_to_buffer(void **buffer, int *size, int word)
{
	append_pair_to_buffer(buffer, size, word % 256, word / 256);
}


void get_pair_from_buffer(void **buffer, int *size, int *lohalf,
	int *hihalf, int offset)
{
	int len, type, data_len;

	len = get_header(buffer, size, &type, &data_len);
	assert(len > 0);

	assert(offset >= 0);
	assert(offset < data_len);

	get_pair(buffer, size, lohalf, hihalf, len / 2 - data_len + offset);
}


void append_header_to_buffer(void **buffer, int *size, int type, int len)
{
	int lflag;

	assert((type & (1 << 13) - 1) == type);

	if (len > 2)
		lflag = 3;
	else
		lflag = len;

	append_word_to_buffer(buffer, size, type | lflag << 14);
	if (lflag == 3)
		append_word_to_buffer(buffer, size, len);
}


/*
**	external
*/


void append_raw_to_buffer(void **buffer, int *size, int type,
	int len, char *raw)
{
	append_header_to_buffer(buffer, size, type, (len + 1) / 2);

	while (len > 1)
	{
		append_pair_to_buffer(buffer, size,
			(int)(unsigned char)raw[0],
			(int)(unsigned char)raw[1]);

		len -= 2;
		raw += 2;
	}

	if (len > 0)
		append_pair_to_buffer(buffer, size,
			(int)(unsigned char)raw[0], 0);
}


int get_type_from_buffer(void **buffer, int *size)
{
	int len, type, data_len;

	len = get_header(buffer, size, &type, &data_len);
	if (len > 0)
		return type;
	else
		return -1;
}


int get_len_from_buffer(void **buffer, int *size)
{
	int len, type, data_len;

	len = get_header(buffer, size, &type, &data_len);
	assert(len > 0);

	return data_len;
}


int get_word_from_buffer(void **buffer, int *size, int offset)
{
	int lohalf, hihalf;

	get_pair_from_buffer(buffer, size, &lohalf, &hihalf, offset);
	return hihalf * 256 + lohalf;
}


void remove_block_from_buffer(void **buffer, int *size)
{
	int len, type, data_len;

	len = get_header(buffer, size, &type, &data_len);
	assert(len > 0);

	remove_from_buffer((char **)buffer, size, len);
}


void remove_block_from_buffer_delay(void **buffer, int *size, int *base)
{
	int len, type, data_len;

	len = get_header(buffer, size, &type, &data_len);
	assert(len > 0);

	*buffer = (char *)*buffer + len;
	*size -= len;
	*base += len;
}


void update_input_buffer(void **buffer, int *size, int base)
{
	if (base > 0)
	{
		*buffer = (char *)*buffer - base;
		*size += base;

		remove_from_buffer((char **)buffer, size, base);
	}
}
