/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <assert.h>

#include "attr.h"
#include "terminal.h"
#include "term_actions.h"
#include "memory.h"
#include "strings.h"

#ifndef lint
static char sccsid[] = "@(#)terminal.c	3.5 2/8/99";
#endif


struct term_state
{
	int width, height;
	int xcursor, ycursor, cursor_wrap;
	int xcursor_save, ycursor_save;
	int color_mode, scroll_top, scroll_bottom;
	char sequence[256];
	int seq_len;
	void *callback_handle;
};


static struct term_state *ts_from_handle(void *handle)
{
	return (struct term_state *)handle;
}


static void do_move_cursor(struct term_state *ts, int x, int y)
{
	callback_move_cursor(ts->callback_handle, x, y);
	ts->xcursor = x;
	ts->ycursor = y;
}


static void scroll_lines_up(struct term_state *ts, int y1, int y2)
{
	if (y1 < y2)
		callback_copy_area(ts->callback_handle,
			0, y1 + 1, -1, y2, 0, y1);

	callback_clear_area(ts->callback_handle, 0, y2, -1, y2);
}


static void scroll_lines_down(struct term_state *ts, int y1, int y2)
{
	if (y1 < y2)
		callback_copy_area(ts->callback_handle,
			0, y1, -1, y2 - 1, 0, y1 + 1);

	callback_clear_area(ts->callback_handle, 0, y1, -1, y1);
}


static void scroll_perhaps(struct term_state *ts)
{
	if (ts->ycursor < ts->scroll_top)
		scroll_lines_down(ts, ts->scroll_top, ts->scroll_bottom),
		do_move_cursor(ts, ts->xcursor, ts->ycursor + 1);
	else if (ts->ycursor > ts->scroll_bottom)
		scroll_lines_up(ts, ts->scroll_top, ts->scroll_bottom),
		do_move_cursor(ts, ts->xcursor, ts->ycursor - 1);
}


static void enter_scroll_region(struct term_state *ts)
{
	if (ts->ycursor < ts->scroll_top)
		do_move_cursor(ts, ts->xcursor, ts->scroll_top);
	else if (ts->ycursor > ts->scroll_bottom)
		do_move_cursor(ts, ts->xcursor, ts->scroll_bottom);
}


static void print_sequence(struct term_state *ts)
{
	char *seq;
	int len;

	for (seq = ts->sequence, len = ts->seq_len;
		len > 0; seq++, len--)
		if (*seq == '\033')
			fprintf(stderr, Term_Esc_Str);
		else if (*seq >= ' ' && *seq <= '~')
			fprintf(stderr, Term_Char_Str, *seq);
		else
			fprintf(stderr, Term_Octal_Str, *seq);

	fprintf(stderr, "\n");
}


static int print_char(struct term_state *ts, int ch)
{
	if ((ch & 0x7f) < ' ')
		return 0;

	if (ts->cursor_wrap)
	{
		term_cursor_abs_horiz((void *)ts, 1);
		term_cursor_rel_vert((void *)ts, 1, Do_Scroll);
	}

	callback_set_char(ts->callback_handle, ts->xcursor, ts->ycursor,
		ch & 0xff | ts->color_mode);

	if (!(ts->cursor_wrap = ts->xcursor == ts->width - 1))
		do_move_cursor(ts, ts->xcursor + 1, ts->ycursor);

	return 1;
}


static int check_sequence(struct term_state *ts, int ch)
{
	int result;

	if (ts->seq_len == sizeof ts->sequence - 1)
	{
		fprintf(stderr, Term_Seq_Too_Long_Err);
		ts->seq_len++;
	}

	if (ts->seq_len >= sizeof ts->sequence)
		return 0;

	ts->sequence[ts->seq_len++] = ch;
	ts->sequence[ts->seq_len] = 0;

	while ((result = exec_sequence((void *)ts,
		ts->sequence)) == Sequence_Invalid &&
		ts->seq_len > 1)
	{
		fprintf(stderr, Term_Seq_Rejected_Err);
		print_sequence(ts);
		ts->sequence[0] = ts->sequence[ts->seq_len - 1];
		ts->seq_len = 1;
	}

	switch (result)
	{
	case Sequence_Incomplete:
		break;

	case Sequence_OK:
		ts->seq_len = 0;
		break;

	case Sequence_Invalid:
		ts->seq_len = 0;
		return print_char(ts, ts->sequence[0]);
	}

	return 0;
}


/*
**	internal and external
*/


void term_cursor_abs_vert(void *handle, int y)
{
	struct term_state *ts;

	ts = ts_from_handle(handle);

	if (y < 1)
		y = 1;

	if (y > ts->height)
		y = ts->height;

	do_move_cursor(ts, ts->xcursor, y - 1);
}


void term_cursor_rel_vert(void *handle, int dy, int scroll)
{
	struct term_state *ts;

	ts = ts_from_handle(handle);

	if (scroll == Dont_Scroll)
		term_cursor_abs_vert(handle, ts->ycursor + 1 + dy);
	else
	{
		enter_scroll_region(ts);
		do_move_cursor(ts, ts->xcursor,
			ts->ycursor + dy);
		scroll_perhaps(ts);
	}
}


void term_cursor_abs_horiz(void *handle, int x)
{
	struct term_state *ts;

	ts = ts_from_handle(handle);

	if (x >= 1 && x <= ts->width)
		do_move_cursor(ts, x - 1, ts->ycursor);
}


void term_clear_to_eol(void *handle)
{
	struct term_state *ts;

	ts = ts_from_handle(handle);

	callback_clear_area(ts->callback_handle,
		ts->xcursor, ts->ycursor, -1, ts->ycursor);
}


/*
**	external only
*/


void term_cursor_rel_horiz(void *handle, int dx)
{
	term_cursor_abs_horiz(handle,
		ts_from_handle(handle)->xcursor + 1 + dx);
}


void term_cursor_tab(void *handle)
{
	struct term_state *ts;
	int x;

	ts = ts_from_handle(handle);

	x = ts->xcursor - ts->xcursor % 8 + 8;
	if (x >= ts->width)
		x = ts->width - 1;

	if (ts->xcursor < x)
		term_cursor_abs_horiz(handle, x + 1);
}


void term_insert_char(void *handle)
{
	struct term_state *ts;

	ts = ts_from_handle(handle);

	callback_copy_area(ts->callback_handle, ts->xcursor, ts->ycursor,
		-1, ts->ycursor, ts->xcursor + 1, ts->ycursor);
	callback_clear_area(ts->callback_handle, ts->xcursor, ts->ycursor,
		ts->xcursor, ts->ycursor);
}


void term_delete_char(void *handle)
{
	struct term_state *ts;

	ts = ts_from_handle(handle);

	callback_copy_area(ts->callback_handle, ts->xcursor + 1, ts->ycursor,
		-1, ts->ycursor, ts->xcursor, ts->ycursor);
}


void term_insert_line(void *handle)
{
	struct term_state *ts;

	ts = ts_from_handle(handle);

	enter_scroll_region(ts);
	scroll_lines_down(ts, ts->ycursor, ts->scroll_bottom);
}


void term_delete_line(void *handle)
{
	struct term_state *ts;

	ts = ts_from_handle(handle);

	enter_scroll_region(ts);
	scroll_lines_up(ts, ts->ycursor, ts->scroll_bottom);
}


void term_clear_to_eod(void *handle)
{
	struct term_state *ts;
	int y;

	ts = ts_from_handle(handle);

	if (ts->xcursor)
		term_clear_to_eol(handle),
		y = ts->ycursor + 1;
	else
		y = ts->ycursor;

	callback_clear_area(ts->callback_handle, 0, y, -1, -1);
}


void term_save_cursor(void *handle)
{
	struct term_state *ts;

	ts = ts_from_handle(handle);

	ts->xcursor_save = ts->xcursor;
	ts->ycursor_save = ts->ycursor;
}


void term_restore_cursor(void *handle)
{
	struct term_state *ts;

	ts = ts_from_handle(handle);

	do_move_cursor(ts, ts->xcursor_save, ts->ycursor_save);
}


int term_get_mode(void *handle)
{
	return ts_from_handle(handle)->color_mode;
}


void term_set_mode(void *handle, int mode)
{
	ts_from_handle(handle)->color_mode = mode;
}


void term_scroll_region(void *handle, int top, int bottom)
{
	struct term_state *ts;

	ts = ts_from_handle(handle);

	if (bottom == BOTTOM_LINE)
		bottom = ts->height;

	if (top >= 1 && top <= bottom && bottom <= ts->height)
	{
		ts->scroll_top = top - 1;
		ts->scroll_bottom = bottom - 1;
	}
}


void term_bell(void *handle)
{
	callback_bell(ts_from_handle(handle)->callback_handle);
}


void write_char(void *handle, int ch)
{
	struct term_state *ts;
	int old_x, old_y;

	ts = ts_from_handle(handle);

	if (ch < 1 || ch > 255)
		return;

	old_x = ts->xcursor;
	old_y = ts->ycursor;

	if (!check_sequence(ts, ch) && (ts->xcursor != old_x ||
		ts->ycursor != old_y))
		ts->cursor_wrap = 0;
}


void *init_term_state(void *handle, int width, int height)
{
	struct term_state *ts;

	ts = mem_alloc(sizeof *ts);

	ts->width = width;
	ts->height = height;

	ts->xcursor_save = 0;
	ts->ycursor_save = 0;
	ts->cursor_wrap = 0;
	ts->color_mode = 0;
	ts->scroll_top = 0;
	ts->scroll_bottom = ts->height - 1;
	ts->seq_len = 0;
	ts->callback_handle = handle;

	do_move_cursor(ts, 0, 0);

	return (void *)ts;
}


void resize_term(void *handle, int width, int height)
{
	struct term_state *ts;
	int x, y;

	ts = ts_from_handle(handle);

	if (ts->xcursor_save >= width)
		ts->xcursor_save = width - 1;
	if (ts->ycursor_save >= height)
		ts->ycursor_save = height - 1;

	if ((x = ts->xcursor) >= width)
		x = width - 1;
	if ((y = ts->ycursor) >= height)
		y = height - 1;

	if (x != ts->xcursor || y != ts->ycursor)
		do_move_cursor(ts, x, y);

	if (ts->scroll_bottom == ts->height - 1)
		ts->scroll_bottom = height - 1;

	if (ts->scroll_top >= height)
		ts->scroll_top = height - 1;
	if (ts->scroll_bottom >= height)
		ts->scroll_bottom = height - 1;

	ts->width = width;
	ts->height = height;
}


void free_term_state(void *handle)
{
	struct term_state *ts;

	ts = ts_from_handle(handle);

	mem_free(ts);
}
