/*
**	Copyright (c) 1995-2000 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions
**	are met:
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <pwd.h>

#include "tty.h"
#include "pty.h"
#include "utmp.h"
#include "memory.h"
#include "strings.h"

#ifndef lint
static char sccsid[] = "@(#)tty.c	3.5 2/8/99";
#endif


struct tty_state
{
	pid_t pid;
	int fd;
};


static struct tty_state *tty_state_from_handle(void *handle)
{
	return (struct tty_state *)handle;
}


/*
**	internal and external
*/


int fd_of_shell(void *handle)
{
	struct tty_state *ts;

	ts = tty_state_from_handle(handle);

	return ts->fd;
}


/*
**	external
*/


int pid_of_shell(void *handle, pid_t pid)
{
	return tty_state_from_handle(handle)->pid == pid;
}


void *fork_shell(int width, int height, print_callback_f print_info,
	char *spawn_path, char **spawn_argv)
{
	int i;
	char ptyname[256];
	struct termios term;
	struct winsize win;
	struct tty_state *ts;

	ts = mem_alloc(sizeof *ts);

	memset(&term, 0, sizeof term);

	term.c_iflag = ICRNL | BRKINT | IXON | IXOFF;
	term.c_oflag = OPOST | ONLCR;
	term.c_cflag = CS8 | CREAD | HUPCL;
	term.c_lflag = ECHO | ECHOE | ECHOK | ISIG | ICANON | IEXTEN;
#ifdef ECHOKE
	term.c_lflag |= ECHOKE;
#endif
#ifdef ECHOCTL
	term.c_lflag |= ECHOCTL;
#endif

	for (i = 0; i < NCCS; i++)
		term.c_cc[i] = -1;

	/* VMIN/VTIME sometimes collide with the other V*,
	   resulting in VEOF = ^A, e.g. */
	term.c_cc[VMIN] = 1;
	term.c_cc[VTIME] = 0;

	term.c_cc[VEOF] = 'D' & 0x3f;
	term.c_cc[VEOL] = -1;
	term.c_cc[VEOL2] = -1;
	term.c_cc[VERASE] = 127;
#ifdef VWERASE
	term.c_cc[VWERASE] = 'W' & 0x3f;
#endif
	term.c_cc[VKILL] = 'U' & 0x3f;
#ifdef VREPRINT
	term.c_cc[VREPRINT] = 'R' & 0x3f;
#endif
#ifdef VRPRNT
	term.c_cc[VRPRNT] = 'R' & 0x3f;
#endif
	term.c_cc[VINTR] = 'C' & 0x3f;
	term.c_cc[VQUIT] = '\\' & 0x3f;
	term.c_cc[VSUSP] = 'Z' & 0x3f;
#ifdef VDSUSP
	term.c_cc[VDSUSP] = 'Y' & 0x3f;
#endif
	term.c_cc[VSTART] = 'Q' & 0x3f;
	term.c_cc[VSTOP] = 'S' & 0x3f;
#ifdef VLNEXT
	term.c_cc[VLNEXT] = 'V' & 0x3f;
#endif
#ifdef VDISCARD
	term.c_cc[VDISCARD] = 'O' & 0x3f;
#endif
#ifdef VFLUSH
	term.c_cc[VFLUSH] = 'O' & 0x3f;
#endif
	cfsetispeed(&term, B9600);
	cfsetospeed(&term, B9600);

	win.ws_col = width;
	win.ws_row = height;
	win.ws_xpixel = 0;
	win.ws_ypixel = 0;

	switch (ts->pid = forkpty(&ts->fd, ptyname, &term, &win))
	{
		case -1:
			mem_free(ts);
			return NULL;

		case 0:
			putenv(Set_Term_Env);

			if (print_info != NULL)
				print_info();

			execvp(spawn_path, spawn_argv);

			fprintf(stderr, Exec_Err, spawn_path);
			exit(1);
	}

	{
		struct passwd *pw;

		pw = getpwuid(getuid());
		utmp_login(ptyname, (pw != NULL) ? pw->pw_name :
			Fallback_Utmpx_User, NULL);
	}

	return ts;
}


void resize_shell(void *handle, int width, int height)
{
	struct winsize win;

	win.ws_col = width;
	win.ws_row = height;
	win.ws_xpixel = 0;
	win.ws_ypixel = 0;

	ioctl(fd_of_shell(handle), TIOCSWINSZ, &win);
}


void close_shell(void *handle)
{
	struct tty_state *ts;

	ts = tty_state_from_handle(handle);

	close(ts->fd);
	mem_free(ts);
}
