/*
**	Copyright (c) 1998, 1999 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions are
**	met:
**
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdlib.h>
#include <X11/Xlib.h>

#include "colormap.h"
#include "event.h"

#ifndef lint
static char sccsid[] = "@(#)colormap.c	1.14 2/8/99";
#endif


typedef struct
{
	int force; /* must always have correct colors? */
	Window top;
	Window client; /* or None */
	Colormap cmap; /* or None */
	int close_down;
} *mapT;


/* shared */
static Display *display;
static Window root_w;

static Atom colormap_proto_atom;
static Colormap default_cmap;

static void *colormap_handle;
static Window colormap_priority_w; /* as requested */
static Colormap current_cmap; /* None before it's first set */
static Window current_w; /* None = unknown */

static int n_maps;
static mapT *maps;


/*ARGSUSED*/
static void e_message(void *handle, XEvent *evP)
{
	if (evP->xclient.message_type == colormap_proto_atom &&
		evP->xclient.format == 32)
		colormap_priority_w = evP->xclient.data.l[0];
}


/*ARGSUSED*/
static void e_enter(void *handle, XEvent *evP)
{
	current_w = evP->xcrossing.window;
}


static void e_colormap(void *handle, XEvent *evP)
{
	mapT map;

	map = handle;
	map->cmap = evP->xcolormap.colormap;
}


static void e_destroy(void *handle, XEvent *evP)
{
	mapT map;

	map = handle;
	if (evP->xdestroywindow.window == map->top ||
		evP->xdestroywindow.window == map->client)
		map->close_down = 1;
}


static void e_reparent(void *handle, XEvent *evP)
{
	mapT map;

	map = handle;
	if (evP->xreparent.window == map->top &&
		evP->xreparent.parent != root_w)
		map->close_down = 1;
}


static Colormap find_colormap(void)
{
	int i;

	for (i = 0; i < n_maps; i++)
		if (maps[i]->top == current_w)
			break;

	if (i < n_maps && maps[i]->force)
		return maps[i]->cmap;

	if (colormap_priority_w == root_w)
		return None;

	if (colormap_priority_w != None)
	{
		int j;

		for (j = 0; j < n_maps; j++)
			if (maps[j]->top == colormap_priority_w ||
				maps[j]->client == colormap_priority_w)
				break;

		if (j < n_maps)
			return maps[j]->cmap;
		else
			/* no longer valid */
			colormap_priority_w = None;
	}

	if (i < n_maps)
		return maps[i]->cmap;
	else
		return None;
}


static Colormap new_colormap(void)
{
	Colormap cmap;

	cmap = find_colormap();
	if (cmap == None)
		return default_cmap;
	else
		return cmap;
}


static void get_current_windowG(void)
{
	Window dummy_w;
	int root_x, root_y, x, y;
	unsigned mask;

	XQueryPointer(display, root_w, &dummy_w, &current_w,
		&root_x, &root_y, &x, &y, &mask);
	if (current_w == None)
		current_w = root_w;
}


/* ----- */


int colormap_has_work(void)
{
	int i;

	for (i = 0; i < n_maps; i++)
		if (maps[i]->close_down)
			return 1;

	return current_w == None || current_cmap != new_colormap();
}


void colormap_do_workG(void)
{
	Colormap new_cmap;
	int i, j;

	for (i = j = 0; i < n_maps; i++)
		if (maps[i]->close_down)
		{
			if (maps[i]->top == current_w)
				current_w = None;

			unregister_event_handlersG(maps[i]);
			free(maps[i]);
		}
		else
			maps[j++] = maps[i];

	n_maps = j;

	if (current_w == None)
		get_current_windowG();

	new_cmap = new_colormap();
	if (current_cmap != new_cmap)
	{
		current_cmap = new_cmap;
		XInstallColormap(display, current_cmap);
	}
}


void colormap_new_windowG(Window w)
{
	int i;
	mapT map;
	XWindowAttributes attr;

	for (i = 0; i < n_maps; i++)
		if (maps[i]->top == w)
			return; /* already known */

	if ((map = malloc(sizeof *map)) == NULL)
		exit(100);

	n_maps++;
	if ((maps = realloc(maps, n_maps * sizeof *maps)) == NULL)
		exit(100);
	maps[n_maps - 1] = map;

	select_eventsG(map, w, EnterWindowMask |
		ColormapChangeMask | StructureNotifyMask);

	XGetWindowAttributes(display, w, &attr);
	map->force = attr.override_redirect;
	map->top = w;
	map->client = None;
	map->cmap = attr.colormap;
	map->close_down = 0;

	register_event_handler(map, w, EnterNotify, e_enter);
	register_event_handler(map, w, ColormapNotify, e_colormap);
	register_event_handler(map, w, DestroyNotify, e_destroy);
	register_event_handler(map, w, ReparentNotify, e_reparent);

	/* might have missed the EnterNotify */
	current_w = None;
}


void colormap_new_clientG(Window top, Window client)
{
	mapT map;
	XWindowAttributes attr;

	if ((map = malloc(sizeof *map)) == NULL)
		exit(100);

	n_maps++;
	if ((maps = realloc(maps, n_maps * sizeof *maps)) == NULL)
		exit(100);
	maps[n_maps - 1] = map;

	select_eventsG(map, top, EnterWindowMask | StructureNotifyMask);
	select_eventsG(map, client, ColormapChangeMask | StructureNotifyMask);

	XGetWindowAttributes(display, client, &attr);
	map->force = 0;
	map->top = top;
	map->client = client;
	map->cmap = attr.colormap;
	map->close_down = 0;

	register_event_handler(map, top, EnterNotify, e_enter);
	register_event_handler(map, top, DestroyNotify, e_destroy);
	register_event_handler(map, top, ReparentNotify, e_reparent);
	register_event_handler(map, client, ColormapNotify, e_colormap);
	register_event_handler(map, client, DestroyNotify, e_destroy);

	/* might have missed the EnterNotify */
	current_w = None;
}


void colormap_initG(Display *set_display, int screen, Window set_root_w)
{
	display = set_display;
	root_w = set_root_w;
	default_cmap = DefaultColormap(display, screen);

	if ((colormap_handle = malloc(1)) == NULL)
		exit(100);

	n_maps = 0;
	if ((maps = malloc(sizeof *maps)) == NULL)
		exit(100);

	colormap_proto_atom = XInternAtom(display,
		"_WWM_COLORMAP_PRIORITY_WINDOW", False);

	colormap_priority_w = None; /* none requested yet */
	current_cmap = None; /* invalid, set it at least once */

	select_eventsG(colormap_handle, root_w, EnterWindowMask);

	register_sendevent_handler(colormap_handle, root_w, ClientMessage,
		e_message);
	register_event_handler(colormap_handle, root_w, EnterNotify, e_enter);

	get_current_windowG();
}
