/*
**	Copyright (c) 1998, 1999 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions are
**	met:
**
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#ifdef DEBUG_EVENTS
#include <stdio.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/extensions/shape.h>

#include "deco.h"
#include "event.h"
#include "resource.h"

#ifndef lint
static char sccsid[] = "@(#)deco.c	1.42 2/8/99";
#endif

#ifdef DEBUG_EVENTS
#ifndef lint
static char sccsid2[] = "@(#)+DEBUG_EVENTS";
#endif
#endif


#define TITLE_H_SPACE 5


struct decoS
{
	Window top; /* None if its gone */
	Window clip_w; /* None if its gone */
	Window title_w; /* None if its gone */
	Window client_w; /* None if its no longer an inferior */

	char *curr_title; /* NULL = damaged */
	char *new_name; /* NULL = fetch again */
	char *new_title; /* truncated to fit into .width pixels */

	int width; /* of client window */
	int full_width, full_height; /* of .top */
	int title_width; /* max{.width, width of .new_title + spacing} */

	/* grabs affect contains_pointer but not ..._focus */
	int contains_pointer;
	int has_focus; /* explicitly, not via PointerRoot */
	int pointer_root_focus; /* whenever the focus is PointerRoot */

	int curr_focus; /* focus indicator in title */

	int close_down, redraw_shadow, do_reshape;
};


/* shared */
static Display *display;
static int screen;

static int has_shapes;
static XFontStruct *font;
static unsigned long bg_pixel;
static GC title_gc, title_bg_gc, top_shadow_gc, bottom_shadow_gc;
static int title_delta;
static int inner_shadow, outer_shadow;
static int border_width; /* excluding the shadows */
static int full_border_width; /* including the shadows */

/* these must match! */
enum { SR_Font, SR_Font_Color, SR_Frame_Color, SR_Top_Color, SR_Bottom_Color };
static struct { char *name, *class, *val; } string_res[] =
{
	{ "wwm.frame.font", "Wwm.Frame.Font" },
	{ "wwm.frame.foreground", "Wwm.Frame.Foreground" },
	{ "wwm.frame.background", "Wwm.Frame.Background" },
	{ "wwm.frame.topShadowColor", "Wwm.Frame.Foreground" },
	{ "wwm.frame.bottomShadowColor", "Wwm.Frame.Foreground" }
};

/* these must match! */
enum { BR_Shape };
static struct { char *name, *class; int val; } bool_res[] =
{
	{ "wwm.frame.useShape", "Wwm.Frame.UseShape" }
};

/* these must match! */
enum { DR_Inner_Shadow, DR_Border, DR_Outer_Shadow };
static struct { char *name, *class; int val; } dim_res[] =
{
	{ "wwm.frame.innerShadowWidth", "Wwm.Frame.ShadowWidth" },
	{ "wwm.frame.borderMargin", "Wwm.Frame.Margin" },
	{ "wwm.frame.outerShadowWidth", "Wwm.Frame.ShadowWidth" },
};


static unsigned long alloc_color(char *name, unsigned long fallback)
{
	XColor color, exact_color;

	if (name != NULL && XAllocNamedColor(display,
		DefaultColormap(display, screen), name, &color, &exact_color))
		return color.pixel;
	else
		return fallback;
}


static char *str_dup(char *str)
{
	char *new;

	if ((new = malloc(strlen(str) + 1)) == NULL)
		exit(100);

	strcpy(new, str);
	return new;
}


static int new_focus(decoT deco)
{
	/* calculate focus indicator */
	return deco->has_focus ||
		deco->pointer_root_focus && deco->contains_pointer;
}


static void redraw_titleG(decoT deco)
{
	char *title;

	if (deco->curr_title != NULL && deco->curr_title != deco->new_title)
		free(deco->curr_title);
	title = deco->curr_title = deco->new_title;

	XDrawImageString(display, deco->title_w, title_gc,
		TITLE_H_SPACE, font->ascent, title, strlen(title));
}


static void redraw_shadowG(decoT deco)
{
	int old_focus, shaped;
	XPoint p[3];

	old_focus = deco->curr_focus;
	deco->curr_focus = new_focus(deco);
	deco->redraw_shadow = 0;

	shaped = deco->title_width != deco->width;


	/* outer shadow */

	if (outer_shadow > 0)
	{
		p[0].x = deco->title_width + 2 * full_border_width - 2;
		p[0].y = 0;
		p[1].x = 0; p[1].y = 0;
		p[2].x = 0; p[2].y = deco->full_height - 2;
		XDrawLines(display, deco->top, top_shadow_gc,
			p, 3, CoordModeOrigin);

		p[0].x = deco->full_width - 1;
		p[0].y = shaped ? title_delta : 0;
		p[1].x = deco->full_width - 1;
		p[1].y = deco->full_height - 1;
		p[2].x = 0; p[2].y = deco->full_height - 1;
		XDrawLines(display, deco->top, bottom_shadow_gc,
			p, 3, CoordModeOrigin);
	}

	if (outer_shadow > 0 && shaped)
	{
		XDrawLine(display, deco->top, top_shadow_gc,
			deco->title_width + 2 * full_border_width - 1,
			title_delta, deco->full_width - 2, title_delta);

		XDrawLine(display, deco->top, bottom_shadow_gc,
			deco->title_width + 2 * full_border_width - 1, 0,
			deco->title_width + 2 * full_border_width - 1,
			title_delta - 1);
	}


	/* inner shadow */

	if (inner_shadow > 0)
	{
		p[0].x = deco->full_width - full_border_width;
		p[0].y = full_border_width - 1 + title_delta;
		p[1].x = deco->full_width - full_border_width;
		p[1].y = deco->full_height - full_border_width;
		p[2].x = full_border_width - 1;
		p[2].y = deco->full_height - full_border_width;
		XDrawLines(display, deco->top, top_shadow_gc,
			p, 3, CoordModeOrigin);

		p[0].x = deco->full_width - full_border_width - 1;
		p[0].y = full_border_width - 1 + title_delta;
		p[1].x = full_border_width - 1;
		p[1].y = full_border_width - 1 + title_delta;
		p[2].x = full_border_width - 1;
		p[2].y = deco->full_height - full_border_width - 1;
		XDrawLines(display, deco->top, bottom_shadow_gc,
			p, 3, CoordModeOrigin);
	}


	/* inner title bar shadow */

	if (inner_shadow > 0 && deco->curr_focus)
	{
		p[0].x = deco->title_width + full_border_width;
		p[0].y = full_border_width - 1;
		p[1].x = deco->title_width + full_border_width;
		p[1].y = title_delta + outer_shadow - inner_shadow;
		p[2].x = full_border_width - 1;
		p[2].y = title_delta + outer_shadow - inner_shadow;
		XDrawLines(display, deco->top, top_shadow_gc,
			p, 3, CoordModeOrigin);

		p[0].x = deco->title_width + full_border_width - 1;
		p[0].y = full_border_width - 1;
		p[1].x = full_border_width - 1;
		p[1].y = full_border_width - 1;
		p[2].x = full_border_width - 1;
		p[2].y = title_delta + outer_shadow - inner_shadow - 1;
		XDrawLines(display, deco->top, bottom_shadow_gc,
			p, 3, CoordModeOrigin);
	}
	else if (inner_shadow > 0 && deco->curr_focus != old_focus)
		XDrawRectangle(display, deco->top, title_bg_gc,
			full_border_width - 1, full_border_width - 1,
			deco->title_width + 1,
			title_delta + 1 - 2 * inner_shadow - border_width);
}


static void reshapeG(decoT deco)
{
	if (!has_shapes)
		return;

	if (deco->width == deco->title_width)
		XShapeCombineMask(display, deco->top, ShapeBounding,
			0, 0, None, ShapeSet);
	else
	{
		XRectangle r[2];

		r[0].x = 0;
		r[0].y = 0;
		r[0].width = deco->title_width + 2 * full_border_width;
		r[0].height = title_delta;

		r[1].x = 0;
		r[1].y = title_delta;
		r[1].width = deco->full_width;
		r[1].height = deco->full_height - title_delta;

		XShapeCombineRectangles(display, deco->top, ShapeBounding,
			0, 0, r, 2, ShapeSet, YXBanded);
	}
}


static void e_client_property(void *handle, XEvent *evP)
{
	decoT deco;

	deco = handle;

	if (evP->xproperty.atom == XA_WM_NAME)
	{
		if (deco->new_name != NULL)
			free(deco->new_name);

		deco->new_name = NULL;
	}
}


/*ARGSUSED*/
static void e_deco_expose(void *handle, XEvent *evP)
{
	decoT deco;

	deco = handle;
	deco->redraw_shadow = 1;
}


/*ARGSUSED*/
static void e_title_expose(void *handle, XEvent *evP)
{
	decoT deco;

	deco = handle;

	if (deco->curr_title != NULL && deco->curr_title != deco->new_title)
		free(deco->curr_title);

	deco->curr_title = NULL; /* damaged */
}


/*ARGSUSED*/
static void e_clip_unmap(void *handle, XEvent *evP)
{
	decoT deco;

	deco = handle;
	deco->close_down = 1;
}


static void e_clip_reparent(void *handle, XEvent *evP)
{
	decoT deco;

	deco = handle;

	/* client moving away? */
	if (evP->xreparent.window == deco->client_w &&
		evP->xreparent.parent != deco->clip_w)
	{
		deco->client_w = None; /* no longer an inferior */
		deco->close_down = 1;
	}
}


static void e_destroy(void *handle, XEvent *evP)
{
	decoT deco;

	deco = handle;

	if (evP->xdestroywindow.window == deco->top)
		deco->top = None; /* gone */
	if (evP->xdestroywindow.window == deco->clip_w)
		deco->clip_w = None; /* gone */
	if (evP->xdestroywindow.window == deco->title_w)
		deco->title_w = None; /* gone */
	if (evP->xdestroywindow.window == deco->client_w)
		deco->client_w = None; /* gone */

	deco->close_down = 1;
}


static void e_deco_focus(void *handle, XEvent *evP)
{
	decoT deco;

	deco = handle;
	if ((evP->xfocus.mode == NotifyNormal ||
		evP->xfocus.mode == NotifyWhileGrabbed) &&
		evP->xfocus.detail != NotifyPointer)
		deco->has_focus = evP->xfocus.type == FocusIn;
}


static void e_client_crossing(void *handle, XEvent *evP)
{
	decoT deco;

	deco = handle;
	if (evP->xcrossing.detail != NotifyInferior)
		deco->contains_pointer = evP->xcrossing.type == EnterNotify;
}


static void e_root_focus(void *handle, XEvent *evP)
{
	decoT deco;

	deco = handle;
	if ((evP->xfocus.mode == NotifyNormal ||
		evP->xfocus.mode == NotifyWhileGrabbed) &&
		evP->xfocus.detail == NotifyPointerRoot)
		deco->pointer_root_focus = evP->xfocus.type == FocusIn;
}


static void recalc_title(decoT deco)
{
	char *title, *p, *q;
	int n, limit;
	int skipped_colon;

	p = deco->new_name;
	n = strlen(p);
	limit = deco->width - 2 * TITLE_H_SPACE;

	skipped_colon = 0;
	while (n > 0 && XTextWidth(font, p, n) > limit)
		if (p[n - 1] == ' ')
			n--;
		else if (p[0] == ' ')
			p++, n--;
		else if (!skipped_colon && (q = memchr(p, ':', n)) != NULL &&
			q[1] == ' ')
		{
			skipped_colon = 1;
			q++;
			n -= q - p;
			p = q;
			while (n > 0 && p[0] == ' ')
				p++, n--;
		}
		else
			n--; /* drop characters */

	if (n > 0)
	{
		title = str_dup(p);
		title[n] = 0;
	}
	else
		title = str_dup("-"); /* even if a - doesn't fit */

	if (deco->new_title != NULL && deco->new_title != deco->curr_title)
		free(deco->new_title);

	if (deco->curr_title != NULL && !strcmp(deco->curr_title, title))
	{
		free(title);
		deco->new_title = deco->curr_title;
	}
	else
		deco->new_title = title;

	deco->title_width = XTextWidth(font, deco->new_title,
		strlen(deco->new_title)) + 2 * TITLE_H_SPACE + 1;
	if (deco->title_width > deco->width || !has_shapes)
		deco->title_width = deco->width;
}


static void reread_nameG(decoT deco)
{
	char *prop, *name;

	XFetchName(display, deco->client_w, &prop);

	if (prop == NULL || *prop == 0)
		name = str_dup("-");
	else
		name = str_dup(prop);

	if (prop != NULL)
		XFree(prop);

	if (deco->new_name != NULL)
		free(deco->new_name);
	deco->new_name = name;
}


static void destroy_decoG(decoT deco)
{
	unregister_event_handlersG(deco);

	if (deco->top != None)
	{
		XDestroyWindow(display, deco->top); /* destroy everything */

		/* forget the event masks */

		if (deco->client_w != None)
			event_forget_window(deco->client_w);
		if (deco->clip_w != None)
			event_forget_window(deco->clip_w);
		if (deco->title_w != None)
			event_forget_window(deco->title_w);
		event_forget_window(deco->top);
	}

	if (deco->curr_title != NULL && deco->curr_title != deco->new_title)
		free(deco->curr_title);
	if (deco->new_title != NULL)
		free(deco->new_title);
	if (deco->new_name != NULL)
		free(deco->new_name);

#ifdef DEBUG_EVENTS
	printf("-Deco h=%p\n", deco);
#endif

	free(deco);
}


/* ----- */


decoT decorate_windowG(Window w, Window root_w, int x, int y,
	int width, int height)
{
	XSetWindowAttributes v;
	decoT deco;

	if ((deco = malloc(sizeof *deco)) == NULL)
		exit(100);

#ifdef DEBUG_EVENTS
	printf("+Deco h=%p w=%#lx\n", deco, (unsigned long)w);
#endif

	deco->client_w = w;
	deco->close_down = 0;
	deco->do_reshape = 0;
	deco->width = width;
	deco->full_width = width + 2 * full_border_width;
	deco->full_height = height + 2 * full_border_width + title_delta;

	deco->contains_pointer = 0;
	deco->has_focus = 0;

	deco->curr_focus = 0;

	deco->curr_title = NULL;
	deco->new_title = NULL;
	deco->new_name = NULL;

	reread_nameG(deco);
	recalc_title(deco);

	v.background_pixel = bg_pixel;
	/* Redirect prevents clip_w configure */
	v.event_mask = ExposureMask | FocusChangeMask |
		SubstructureRedirectMask | StructureNotifyMask;
	deco->top = XCreateWindow(display, root_w, x - full_border_width,
		y - full_border_width - title_delta,
		width + 2 * full_border_width,
		height + 2 * full_border_width + title_delta, 0,
		CopyFromParent /* depth */,
		InputOutput, CopyFromParent /* visual */,
		CWBackPixel | CWEventMask, &v);
	created_with_event_maskG(deco, deco->top, v.event_mask);

	reshapeG(deco);

	v.background_pixmap = ParentRelative;
	v.event_mask = ExposureMask | StructureNotifyMask;
	deco->title_w = XCreateWindow(display, deco->top, full_border_width,
		full_border_width, deco->title_width,
		title_delta - 2 * inner_shadow - border_width, 0,
		CopyFromParent /* depth */,
		InputOutput, CopyFromParent /* visual */,
		CWBackPixmap | CWEventMask, &v);
	created_with_event_maskG(deco, deco->title_w, v.event_mask);

	v.background_pixmap = ParentRelative;
	v.event_mask = SubstructureNotifyMask | StructureNotifyMask;
	deco->clip_w = XCreateWindow(display, deco->top, full_border_width,
		full_border_width + title_delta,
		width, height, 0, CopyFromParent /* depth */,
		InputOutput, CopyFromParent /* visual */,
		CWBackPixmap | CWEventMask, &v);
	created_with_event_maskG(deco, deco->clip_w, v.event_mask);

	register_event_handler(deco, deco->top, FocusIn, e_deco_focus);
	register_event_handler(deco, deco->top, FocusOut, e_deco_focus);

	select_eventsG(deco, w, PropertyChangeMask |
		EnterWindowMask | LeaveWindowMask);
	register_event_handler(deco, w, PropertyNotify, e_client_property);
	register_event_handler(deco, w, EnterNotify, e_client_crossing);
	register_event_handler(deco, w, LeaveNotify, e_client_crossing);

	select_eventsG(deco, root_w, FocusChangeMask);
	register_event_handler(deco, root_w, FocusIn, e_root_focus);
	register_event_handler(deco, root_w, FocusOut, e_root_focus);

	{
		Window focus_w;
		int revert;

		XGetInputFocus(display, &focus_w, &revert);
		deco->pointer_root_focus = focus_w == PointerRoot;
	}

	/* pretend its ok, wait for Expose */
	deco->curr_title = deco->new_title;
	deco->redraw_shadow = 0;

	XMapWindow(display, deco->title_w);
	XMapWindow(display, deco->clip_w);

	register_event_handler(deco, deco->top, Expose, e_deco_expose);
	register_event_handler(deco, deco->title_w, Expose,
		e_title_expose);
	register_event_handler(deco, deco->clip_w, UnmapNotify, e_clip_unmap);
	register_event_handler(deco, deco->clip_w, ReparentNotify,
		e_clip_reparent);

	register_event_handler(deco, deco->top, DestroyNotify, e_destroy);
	register_event_handler(deco, deco->title_w, DestroyNotify, e_destroy);
	register_event_handler(deco, deco->clip_w, DestroyNotify, e_destroy);
	return deco;
}


Window deco_toplevel(decoT deco)
{
	return deco->top;
}


Window get_clip_window(decoT deco)
{
	return deco->clip_w;
}


void translate_coords(XWindowChanges *chP)
{
	chP->x += full_border_width;
	chP->y += full_border_width + title_delta;

	if (chP->width <= 2 * full_border_width)
		chP->width = 1;
	else
		chP->width -= 2 * full_border_width;

	if (chP->height <= 2 * full_border_width + title_delta)
		chP->height = 1;
	else
		chP->height -= 2 * full_border_width + title_delta;
}


int deco_has_work(decoT deco)
{
	return deco->close_down ||
		deco->new_name == NULL ||
		deco->curr_title != deco->new_title ||
		deco->curr_focus != new_focus(deco) ||
		deco->redraw_shadow ||
		deco->do_reshape;
}


int deco_do_workG(decoT deco)
{
	if (deco->close_down)
	{
		if (deco->client_w != None)
		{
			reparent_clientG(deco->client_w);
			deco->client_w = None;
		}

		destroy_decoG(deco);
		return 0; /* gone */
	}

	if (deco->new_name == NULL)
	{
		reread_nameG(deco);
		deco->do_reshape = 1;
	}

	if (deco->do_reshape)
	{
		recalc_title(deco);
		XResizeWindow(display, deco->title_w, deco->title_width,
			title_delta - 2 * inner_shadow - border_width);
		reshapeG(deco);
		deco->do_reshape = 0;

		XClearWindow(display, deco->top);
		deco->redraw_shadow = 1;
	}

	if (deco->curr_title != deco->new_title)
		redraw_titleG(deco);

	if (deco->redraw_shadow || deco->curr_focus != new_focus(deco))
		redraw_shadowG(deco);

	return 1; /* still alive */
}


void configure_decoG(decoT deco, unsigned mask, XWindowChanges *chP)
{
	XWindowChanges ch;

	if (mask & (CWWidth | CWHeight))
		deco->do_reshape = 1;

	if (mask & CWWidth)
	{
		deco->width = chP->width;
		deco->full_width = chP->width + 2 * full_border_width;

		ch.width = deco->title_width;
#ifdef DEBUG_EVENTS
		printf("h=%p serial=%d w=%#lx mask=%#x\n",
			deco, NextRequest(display),
			(unsigned long)deco->title_w, CWWidth);
#endif
		XConfigureWindow(display, deco->title_w, CWWidth, &ch);
	}

	if (mask & CWHeight)
		deco->full_height = chP->height +
			2 * full_border_width + title_delta;

	memcpy(&ch, chP, sizeof ch);

	ch.x -= full_border_width;
	ch.y -= full_border_width + title_delta;

	ch.width += 2 * full_border_width;
	ch.height += 2 * full_border_width + title_delta;

#ifdef DEBUG_EVENTS
	printf("h=%p serial=%d w=%#lx mask=%#x\n", deco, NextRequest(display),
		(unsigned long)deco->top, mask);
#endif
	XConfigureWindow(display, deco->top, mask, &ch);

	mask &= CWWidth | CWHeight;
	if (mask)
	{
#ifdef DEBUG_EVENTS
		printf("h=%p serial=%d w=%#lx mask=%#x\n",
			deco, NextRequest(display),
			(unsigned long)deco->clip_w, mask);
#endif
		XConfigureWindow(display, deco->clip_w, mask, chP);
	}
}


void deco_init(Display *set_display, int set_screen, Window root_w)
{
	XGCValues v;
	unsigned long black;
	int event_base, error_base;
	int i;

	display = set_display;
	screen = set_screen;

	for (i = 0; i < sizeof string_res / sizeof *string_res; i++)
		string_res[i].val = resource_lookup_string(string_res[i].name,
			string_res[i].class, NULL);

	for (i = 0; i < sizeof bool_res / sizeof *bool_res; i++)
		bool_res[i].val = resource_lookup_bool(bool_res[i].name,
			bool_res[i].class, False);

	for (i = 0; i < sizeof dim_res / sizeof *dim_res; i++)
		dim_res[i].val = resource_lookup_dimension(dim_res[i].name,
			dim_res[i].class, 0);

	has_shapes = bool_res[BR_Shape].val &&
		XShapeQueryExtension(display, &event_base, &error_base);

	black = BlackPixel(display, screen);
	bg_pixel = alloc_color(string_res[SR_Frame_Color].val,
		WhitePixel(display, screen));

	font = NULL;
	if (string_res[SR_Font].val != NULL)
		font = XLoadQueryFont(display, string_res[SR_Font].val);
	if (font == NULL)
		font = XLoadQueryFont(display, "fixed");
	if (font == NULL)
		exit(1);

	inner_shadow = dim_res[DR_Inner_Shadow].val;
	outer_shadow = dim_res[DR_Outer_Shadow].val;
	border_width = dim_res[DR_Border].val;

	full_border_width = inner_shadow + outer_shadow + border_width;
	title_delta = font->ascent + font->descent +
		2 * inner_shadow + border_width;

	v.foreground = alloc_color(string_res[SR_Font_Color].val, black);
	v.background = bg_pixel;
	v.font = font->fid;
	v.graphics_exposures = False;
	title_gc = XCreateGC(display, root_w, GCForeground |
		GCBackground | GCFont | GCGraphicsExposures, &v);

	v.foreground = bg_pixel;
	v.graphics_exposures = False;
	title_bg_gc = XCreateGC(display, root_w,
		GCForeground | GCGraphicsExposures, &v);

	v.foreground = alloc_color(string_res[SR_Top_Color].val, black);
	v.graphics_exposures = False;
	top_shadow_gc = XCreateGC(display, root_w,
		GCForeground | GCGraphicsExposures, &v);

	v.foreground = alloc_color(string_res[SR_Bottom_Color].val, black);
	v.graphics_exposures = False;
	bottom_shadow_gc = XCreateGC(display, root_w,
		GCForeground | GCGraphicsExposures, &v);
}
