/*
**	Copyright (c) 1998, 1999 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions are
**	met:
**
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#ifdef DEBUG_EVENTS
#include <stdio.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>

#include "event.h"

#ifndef lint
static char sccsid[] = "@(#)event.c	1.13 2/8/99";
#endif

#ifdef DEBUG_EVENTS
#ifndef lint
static char sccsid2[] = "@(#)+DEBUG_EVENTS";
#endif
#endif


typedef struct
{
	void *handle;
	Window w;
	long mask;
} maskS;

typedef struct
{
	void *handle;
	Window w;
	int type;
	Bool send_event;
	handlerF f;
	int pending;
} handlerS;


/* shared */
static Display *display;

static int n_handlers;
static handlerS *handlers;

static int n_masks;
static maskS *masks;


static long get_mask(Window w)
{
	int i;
	long mask;

	mask = 0;
	for (i = 0; i < n_masks; i++)
		if (masks[i].w == w)
			mask |= masks[i].mask;

	/* we always add StructureNotifyMask to non-empty masks
	   so we can detect the window disappearing */
	if (mask != 0)
		mask |= StructureNotifyMask;

	/* XXX should add StructureNotifyMask for windows with
	   event handlers, too */

	return mask;
}


static void update_maskG(void *handle, Window w,
	long event_mask, long curr_mask)
{
	int i;
	long new_mask;

	for (i = 0; i < n_masks; i++)
		if (masks[i].handle == handle && masks[i].w == w)
			break;

	/* not found but need it? */
	if (i >= n_masks && event_mask != 0)
	{
		n_masks++;
		if ((masks = realloc(masks, n_masks * sizeof *masks)) == NULL)
			exit(100);

		i = n_masks - 1;
		masks[i].handle = handle;
		masks[i].w = w;
		masks[i].mask = event_mask;
	}
	/* found and need it? */
	else if (i < n_masks && event_mask != 0)
		masks[i].mask = event_mask;
	/* found but don't need it? */
	else if (i < n_masks && event_mask == 0)
	{
		n_masks--;
		if (i < n_masks)
			memmove(&masks[i], &masks[i + 1],
				(n_masks - i) * sizeof *masks);
	}

	new_mask = get_mask(w);

	/* need to change it? */
	if (new_mask != curr_mask)
	{
#ifdef DEBUG_EVENTS
		printf("h=%p serial=%d w=%#x: %#x -> %#x\n", handle,
			NextRequest(display), w, curr_mask, new_mask);
#endif
		XSelectInput(display, w, new_mask);
	}
}


void created_with_event_maskG(void *handle, Window w, long event_mask)
{
	/* might add StructureNotifyMask */
	update_maskG(handle, w, event_mask, event_mask);
}


void select_eventsG(void *handle, Window w, long event_mask)
{
	update_maskG(handle, w, event_mask, get_mask(w));
}


void register_event_handler(void *handle, Window w, int type, handlerF handler)
{
	handlerS *h;

	n_handlers++;
	if ((handlers = realloc(handlers,
		n_handlers * sizeof *handlers)) == NULL)
		exit(100);

	h = &handlers[n_handlers - 1];

	h->handle = handle;
	h->w = w;
	h->type = type;
	h->send_event = False;
	h->f = handler;
	h->pending = 0;
}


void register_sendevent_handler(void *handle, Window w,
	int type, handlerF handler)
{
	handlerS *h;

	n_handlers++;
	if ((handlers = realloc(handlers,
		n_handlers * sizeof *handlers)) == NULL)
		exit(100);

	h = &handlers[n_handlers - 1];

	h->handle = handle;
	h->w = w;
	h->type = type;
	h->send_event = True;
	h->f = handler;
	h->pending = 0;
}


void unregister_event_handlersG(void *handle)
{
	int i, j;

	for (i = j = 0; i < n_handlers; i++)
		if (handlers[i].handle != handle)
		{
			memcpy(&handlers[j], &handlers[i], sizeof *handlers);
			j++;
		}

	n_handlers = j;

	for (;;)
	{
		for (i = 0; i < n_masks && masks[i].handle != handle; i++);
		if (i >= n_masks)
			break;

		/* this will remove the mask entry */
		select_eventsG(handle, masks[i].w, 0);
	}
}


void event_forget_window(Window w)
{
	int i, j;

	/* forget all masks for this window */
	for (i = j = 0; i < n_masks; i++)
		if (masks[i].w != w)
		{
			memcpy(&masks[j], &masks[i], sizeof *masks);
			j++;
		}

	n_masks = j;
}


void handle_event(XEvent *evP)
{
	int i;

	for (i = 0; i < n_handlers; i++)
	{
		handlerS *h;

		h = &handlers[i];
		h->pending = h->w == evP->xany.window &&
			h->type == evP->xany.type &&
			h->send_event == evP->xany.send_event;
	}

#ifdef DEBUG_EVENTS
	printf("\tw=%#x t=%d\n", evP->xany.window, evP->xany.type);
#endif

	if (evP->xany.type == DestroyNotify && !evP->xany.send_event)
		event_forget_window(evP->xdestroywindow.window);

	for (;;)
	{
		for (i = 0; i < n_handlers && !handlers[i].pending; i++);
		if (i >= n_handlers)
			return;

		handlers[i].pending = 0;

#ifdef DEBUG_EVENTS
		printf("h=%p w=%#x t=%d\n", handlers[i].handle,
			evP->xany.window, evP->xany.type);
#endif

		/* might add or remove handlers */
		handlers[i].f(handlers[i].handle, evP);
	}
}


void event_init(Display *set_display)
{
	display = set_display;

	n_handlers = 0;
	if ((handlers = malloc(sizeof *handlers)) == NULL)
		exit(100);

	n_masks = 0;
	if ((masks = malloc(sizeof *masks)) == NULL)
		exit(100);
}
