/*
**	Copyright (c) 1998, 1999 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions are
**	met:
**
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#ifndef lint
static char sccsid[] = "@(#)keytools.c	1.17 2/8/99";
#endif


/* Digital Unix has a copy of this in <unistd.h> */
extern char **environ;


typedef struct
{
	char *name;
	char **tool;
	int keycode;
	unsigned mod;
	Time last_t;
} keyT;


static char *me;
static Display *display;
static Window root_w, master_w;

static XErrorHandler old_handler;
static int request_failed;

static struct { char *name; unsigned mod; } modifiers[] =
{
	{ "Meta", 0 },
	{ "Alt", 0 },
	{ "Shift", ShiftMask },
	{ "Lock", LockMask },
	{ "Control", ControlMask },
	{ "Mod1", Mod1Mask },
	{ "Mod2", Mod2Mask },
	{ "Mod3", Mod3Mask },
	{ "Mod4", Mod4Mask },
	{ "Mod5", Mod5Mask }
};

static int n_keys;
static keyT *keys;


/* XSH5 requires SA_NOCLDWAIT, but FreeBSD < 3.0 (and probably others)
   doesn't provide it */

#ifndef SA_NOCLDWAIT
static void sigchld_handler(int sig)
{
	while (waitpid(-1, NULL, WNOHANG) > 0);
}
#endif


static void spawn(char **av)
{
	if (vfork())
		return; /* error or parent */

	execvp(av[0], av);
	_exit(255);
}


/* name is R/W */
static int parse_key_name(char *name, int *keycodeP, unsigned *modP)
{
	char *p;
	unsigned mod;
	KeySym sym;
	KeyCode keycode;

	mod = 0;
	while ((p = strchr(name, ' ')) != NULL)
	{
		int len, j;

		len = p - name;

		for (j = sizeof modifiers / sizeof *modifiers;
			j-- > 0;)
			if (strlen(modifiers[j].name) == len &&
				!strncmp(modifiers[j].name, name, len))
				break;

		if (j < 0)
		{
			name[len] = 0;
			fprintf(stderr, "%s: unknown modifier `%s'", me, name);
			name[len] = ' ';
			return 0;
		}

		if (modifiers[j].mod == 0)
		{
			name[len] = 0;
			fprintf(stderr, "%s: inactive modifier `%s'",
				me, name);
			name[len] = ' ';
			return 0;
		}

		mod |= modifiers[j].mod;
		name = name + len + 1;
	}

	*modP = mod;

	sym = XStringToKeysym(name);
	if (sym == NoSymbol)
	{
		fprintf(stderr, "%s: unknown key name `%s'", me, name);
		return 0;
	}

	keycode = XKeysymToKeycode(display, sym);
	if (keycode == 0)
	{
		fprintf(stderr, "%s: inactive key `%s'", me, name);
		return 0;
	}

	*keycodeP = keycode;
	return 1;
}


static char **parse_tool(char *tool)
{
	char **av, *p;
	int n, len;

	n = 0;
	if ((av = malloc(sizeof *av)) == NULL)
		exit(1);

	while ((p = strchr(tool, ' ')) != NULL)
	{
		len = p - tool;
		n++;
		if ((av = realloc(av, n * sizeof *av)) == NULL ||
			(av[n - 1] = malloc(len + 1)) == NULL)
			exit(1);

		strncpy(av[n - 1], tool, len);
		av[n - 1][len] = 0;

		tool += len + 1;
	}

	len = strlen(tool);
	n += 2;
	if ((av = realloc(av, n * sizeof *av)) == NULL ||
		(av[n - 2] = malloc(len + 1)) == NULL)
		exit(1);
	strcpy(av[n - 2], tool);
	av[n - 1] = NULL;

	return av;
}


static char *sanitize_line(char *line)
{
	char *l, *p;
	unsigned char *u;

	if ((l = malloc(strlen(line) + 1)) == NULL)
		exit(1);
	strcpy(l, line);
	line = l;

	for (u = (unsigned char *)l; *u; u++)
		if (*u < 0x20 || *u > 0x7e && *u < 0xa0)
			*u = ' ';

	while ((p = strstr(line, "  ")) != NULL)
		memmove(p + 1, p + 2, strlen(p + 2) + 1);

	if (line[0] == ' ')
		memmove(line, line + 1, strlen(line + 1) + 1);

	if (line[0] == 0)
		return line;

	if (line[strlen(line) - 1] == ' ')
		line[strlen(line) - 1] = 0;

	return line;
}


static void add_key(char *line)
{
	char *p, *name, *tool;
	int keycode;
	unsigned mod;

	line = sanitize_line(line);
	if (line[0] == 0 || line[0] == '!')
	{
		free(line);
		return; /* ignore empty and comment lines */
	}

	if ((p = strchr(line, ':')) == NULL || p == line || p[1] == 0)
	{
		fprintf(stderr, "%s: can't parse `%s'\n", me, line);
		free(line);
		return;
	}

	*p = 0;
	if (p[-1] == ' ')
		p[-1] = 0;
	if (p[1] == ' ')
		p++;

	name = line;
	tool = p + 1;

	if (!parse_key_name(name, &keycode, &mod))
	{
		fprintf(stderr, ", ignoring `%s'\n", name);
		free(line);
		return;
	}

	n_keys++;
	if ((keys = realloc(keys, n_keys * sizeof *keys)) == NULL)
		exit(1);

	keys[n_keys - 1].name = name;
	keys[n_keys - 1].tool = parse_tool(tool);
	keys[n_keys - 1].keycode = keycode;
	keys[n_keys - 1].mod = mod;
}


static int grab_error_handler(Display *d, XErrorEvent *evP)
{
	if (evP->error_code == BadAccess)
	{
		/* soft error */
		request_failed = 1;
		return 0;
	}

	return old_handler(d, evP);
}


int main(int argc, char **argv)
{
	int i, j, n;
	char **env0, **env1;
	char w_env[512], x_env[512], y_env[512];

	if (argc < 1)
		me = "(unknown)";
	else if ((me = strrchr(argv[0], '/')) == NULL)
		me = argv[0];
	else if (*++me == 0)
		me = argv[0];

	if (argc != 1)
	{
		fprintf(stderr, "usage: %s\n", me);
		exit(10);
	}

	{
		struct sigaction s;

#ifdef SA_NOCLDWAIT
		s.sa_handler = SIG_DFL;
		s.sa_flags = SA_NOCLDWAIT;
#else
		s.sa_handler = sigchld_handler;
		s.sa_flags = 0;
#endif

		sigemptyset(&s.sa_mask);
		sigaction(SIGCHLD, &s, NULL);
	}

	for (n = 0; environ[n] != NULL; n++);
	if ((env0 = malloc((n + 1) * sizeof *env0)) == NULL ||
		(env1 = malloc((n + 4) * sizeof *env1)) == NULL)
	{
		fprintf(stderr, "%s: out of memory\n", me);
		exit(100);
	}

	for (i = j = 0; i < n; i++)
		if (strncmp(environ[i], "KEYTOOLS_", 9))
			env0[j++] = environ[i];
	n = j;
	env0[n] = NULL;

	memcpy(env1, env0, n * sizeof *env1);
	env1[n] = w_env;
	env1[n + 1] = x_env;
	env1[n + 2] = y_env;
	env1[n + 3] = NULL;

	display = XOpenDisplay(NULL);
	if (display == NULL)
	{
		fprintf(stderr, "%s: can't open display %s\n",
			me, XDisplayName(NULL));
		exit(1);
	}

	root_w = DefaultRootWindow(display);
	XSelectInput(display, root_w, KeyPressMask | KeyReleaseMask);

	{
		XTextProperty prop, *propP;
		char *name;
		XSetWindowAttributes v;

		v.event_mask = StructureNotifyMask;
		v.override_redirect = True;
		master_w = XCreateWindow(display, root_w, -10, -10, 5, 5, 0,
			CopyFromParent, CopyFromParent, CopyFromParent,
			CWEventMask | CWOverrideRedirect, &v);

		name = "keytools";
		if (XStringListToTextProperty(&name, 1, &prop))
			propP = &prop;
		else
			propP = NULL;

		XSetWMProperties(display, master_w, propP, propP, argv, argc,
			NULL, NULL, NULL);

		if (propP != NULL)
			XFree(propP->value);
	}

	{
		XModifierKeymap *mods;
		KeyCode keycode;
		KeySym sym;

		mods = XGetModifierMapping(display);

		for (i = 0; i < 8; i++)
			for (j = 0; j < mods->max_keypermod; j++)
			{
				keycode = mods->modifiermap[j +
					mods->max_keypermod * i];
				if (keycode == 0)
					continue;

				sym = XKeycodeToKeysym(display, keycode, 0);
				if (sym == XK_Meta_L || sym == XK_Meta_R)
					modifiers[0].mod = 1 << i;
				else if (sym == XK_Alt_L || sym == XK_Alt_R)
					modifiers[1].mod = 1 << i;
			}

		XFreeModifiermap(mods);
	}

	n_keys = 0;
	if ((keys = malloc(sizeof *keys)) == NULL)
		exit(1);

	{
		char *home, *fname;
		char buffer[8192];
		FILE *f;

		home = getenv("HOME");
		if (home == NULL)
		{
			char *user;
			struct passwd *pw;

			user = getenv("USER");
			if (user != NULL && (pw = getpwnam(user)) != NULL)
				home = pw->pw_dir;
			else if ((pw = getpwuid(getuid())) != NULL)
				home = pw->pw_dir;
			else
				home = ""; /* -> /.file */
		}

		if ((fname = malloc(strlen(home) + 40)) == NULL)
			exit(1);
		strcpy(fname, home);
		strcat(fname, "/.keytools");

		if ((f = fopen(fname, "r")) == NULL)
		{
			fprintf(stderr, "%s: can't open %s\n", me, fname);
			exit(1);
		}

		while (fgets(buffer, sizeof buffer, f) != NULL)
		{
			int ch;

			if (buffer[0] == 0)
				break;

			if (buffer[strlen(buffer) - 1] == '\n')
			{
				buffer[strlen(buffer) - 1] = 0;
				add_key(buffer);
				continue;
			}

			while ((ch = fgetc(f)) != EOF && ch != 'n');
			if (ch == EOF)
				break;
		}

		fclose(f);
	}

	if (n_keys < 1)
	{
		fprintf(stderr, "%s: no keys to handle\n", me);
		exit(1);
	}

	XSync(display, False);
	old_handler = XSetErrorHandler(grab_error_handler);

	for (i = 0; i < n_keys; i++)
	{
		keyT *kP;

		kP = &keys[i];
		kP->last_t = 0; /* don't know how to get the server time */

		request_failed = 0;
		XGrabKey(display, kP->keycode, kP->mod, root_w, False,
			GrabModeAsync, GrabModeAsync);
		XSync(display, False);

		if (request_failed)
			fprintf(stderr, "%s: can't grab %s\n", me, kP->name);
	}

	XSetErrorHandler(old_handler);

	for (;;)
	{
		XEvent ev;

		XNextEvent(display, &ev);
		if (ev.xany.send_event)
			continue;

		if (ev.type == DestroyNotify &&
			ev.xdestroywindow.window == master_w)
			break;

		if ((ev.type == KeyPress || ev.type == KeyRelease) &&
			ev.xkey.window == root_w)
		{
			int too_fast;

			for (i = 0; i < n_keys; i++)
				if (keys[i].keycode == ev.xkey.keycode &&
					keys[i].mod == ev.xkey.state)
					break;
			if (i >= n_keys)
				continue;

			too_fast = ev.xkey.time - keys[i].last_t < 100U;
			keys[i].last_t = ev.xkey.time;

			if (too_fast)
				continue; /* ignore repeating key */

			if (ev.type == KeyPress && !too_fast)
			{
				/* this is a key press and the last key press
				   or release is older than 100ms */

				if (ev.xkey.same_screen)
				{
					unsigned long w;

					w = ev.xkey.subwindow;
					sprintf(w_env, "KEYTOOLS_WINDOW=%#lx",
						w);
					sprintf(x_env, "KEYTOOLS_X=%d",
						ev.xkey.x_root);
					sprintf(y_env, "KEYTOOLS_Y=%d",
						ev.xkey.y_root);

					environ = env1;
					spawn(keys[i].tool);
				}
				else
				{
					environ = env0;
					spawn(keys[i].tool);
				}
			}
		}
	}

	XCloseDisplay(display);
	return 0;
}
