/*
**	Copyright (c) 1998, 1999 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions are
**	met:
**
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#ifdef DEBUG_EVENTS
#include <stdio.h>
#endif

#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "wmstate.h"
#include "event.h"

#ifndef lint
static char sccsid[] = "@(#)wmstate.c	1.12 2/8/99";
#endif

#ifdef DEBUG_EVENTS
#ifndef lint
static char sccsid2[] = "@(#)+DEBUG_EVENTS";
#endif
#endif


#define MAX2(a, b) ((a) > (b) ? (a) : (b))
#define MAX3(a, b, c) (MAX2((a), MAX2((b), (c))))

#define NO_STATE (MAX3(WithdrawnState, NormalState, IconicState) + 1)


typedef struct
{
	Window w, parent, last_parent;
	int state; /* WithdrawnState, NormalState, IconicState */
	int last_state; /* any of the above or NO_STATE */
	int state_request; /* NormalState, IconicState, NO_STATE */
	int in_save_set;
	int close_down;
} *clientT;


/* shared */
static Display *display;
static Window root_w;

static Atom wmstate_atom, state_proto_atom;

static void *wmstate_handle;

static int n_clients;
static clientT *clients;


static void e_destroy(void *handle, XEvent *evP)
{
	clientT c;

	c = handle;
	if (evP->xdestroywindow.window == c->w)
		c->close_down = 1;
}


static void e_reparent(void *handle, XEvent *evP)
{
	clientT c;

	c = handle;
	if (evP->xreparent.window == c->w)
	{
		if (evP->xreparent.parent != c->parent)
			c->close_down = 1;
		else
			c->last_parent = c->parent;
	}
}


/*ARGSUSED*/
static void e_root_unmap(void *handle, XEvent *evP)
{
	Window w;
	int i;

	w = evP->xunmap.window;
	for (i = 0; i < n_clients; i++)
		if (clients[i]->w == w)
		{
			clients[i]->state = WithdrawnState;
			return;
		}
}


/*ARGSUSED*/
static void e_client_unmap(void *handle, XEvent *evP)
{
	clientT c;

	c = handle;
	if (c->parent == c->last_parent && c->state == NormalState)
		c->state = WithdrawnState;
}


static void e_message(void *handle, XEvent *evP)
{
	clientT c;

	c = handle;
	if (evP->xclient.window == c->w &&
		evP->xclient.message_type == state_proto_atom &&
		evP->xclient.format == 32 &&
		(evP->xclient.data.l[0] == NormalState ||
		evP->xclient.data.l[0] == IconicState))
		c->state_request = evP->xclient.data.l[0];
}


/* ----- */


int wmstate_has_work(void)
{
	int i;

	for (i = 0; i < n_clients; i++)
		if (clients[i]->close_down ||
			clients[i]->state != clients[i]->last_state ||
			clients[i]->state == WithdrawnState &&
				clients[i]->in_save_set ||
			clients[i]->state_request != NO_STATE)
			return 1;

	return 0;
}


void wmstate_do_workG(void)
{
	int i, j;

	for (i = j = 0; i < n_clients; i++)
		if (clients[i]->close_down)
		{
#ifdef DEBUG_EVENTS
			printf("-WMState h=%p w=%#lx\n", clients[i],
				(unsigned long)clients[i]->w);
#endif

			unregister_event_handlersG(clients[i]);
			free(clients[i]);
		}
		else
			clients[j++] = clients[i];

	n_clients = j;

	for (i = 0; i < n_clients; i++)
	{
		if (clients[i]->state_request == clients[i]->state)
			clients[i]->state_request = NO_STATE;

		if (clients[i]->state_request != NO_STATE)
		{
			clients[i]->state = clients[i]->state_request;
			clients[i]->state_request = NO_STATE;

			switch (clients[i]->state)
			{
			case NormalState:
				deiconify_windowG(clients[i]->w);
				return;

			case IconicState:
				iconify_windowG(clients[i]->w);
				return;

			default:
				abort();
			}
		}

		if (clients[i]->state != clients[i]->last_state)
		{
			unsigned long data[2];

			clients[i]->last_state = clients[i]->state;

			data[0] = clients[i]->state;
			data[1] = None; /* icon window */

			XChangeProperty(display, clients[i]->w, wmstate_atom,
				wmstate_atom, 32, PropModeReplace,
				(void *)data, 2);
		}

		if (clients[i]->state == WithdrawnState &&
			clients[i]->in_save_set)
		{
			XRemoveFromSaveSet(display, clients[i]->w);
			clients[i]->in_save_set = 0;
		}
	}
}


void wmstate_new_windowG(Window w, int default_state)
{
	int i;
	clientT c;

	for (i = 0; i < n_clients; i++)
		if (clients[i]->w == w)
			return; /* already known */

	if ((c = malloc(sizeof *c)) == NULL)
		exit(100);

	n_clients++;
	if ((clients = realloc(clients, n_clients * sizeof *clients)) == NULL)
		exit(100);
	clients[n_clients - 1] = c;

#ifdef DEBUG_EVENTS
	printf("+WMState h=%p w=%#lx\n", c, (unsigned long)w);
#endif

	select_eventsG(c, w, StructureNotifyMask);

	c->w = w;
	c->parent = c->last_parent = root_w;
	c->state = default_state; /* NormalState or WithdrawnState */
	c->last_state = c->state_request = NO_STATE;
	c->in_save_set = 0;
	c->close_down = 0;

	register_event_handler(c, w, DestroyNotify, e_destroy);
	register_event_handler(c, w, ReparentNotify, e_reparent);
	register_event_handler(c, w, UnmapNotify, e_client_unmap);
	register_sendevent_handler(c, w, ClientMessage, e_message);
}


void wmstate_will_decorateG(Window w)
{
	int i;

	for (i = 0; i < n_clients; i++)
		if (clients[i]->w == w)
		{
			clients[i]->state = NormalState;
			clients[i]->in_save_set = 1;
			XAddToSaveSet(display, w);
			return;
		}
}


void wmstate_reparent(Window w, Window parent)
{
	int i;

	for (i = 0; i < n_clients; i++)
		if (clients[i]->w == w)
		{
			clients[i]->parent = parent;
			return;
		}
}


int wmstate_was_iconic(Window w)
{
	int i;

	for (i = 0; i < n_clients; i++)
		if (clients[i]->w == w)
			return clients[i]->last_state == IconicState;

	return 0;
}


void wmstate_initG(Display *set_display, Window set_root_w)
{
	display = set_display;
	root_w = set_root_w;

	if ((wmstate_handle = malloc(1)) == NULL)
		exit(100);

	n_clients = 0;
	if ((clients = malloc(sizeof *clients)) == NULL)
		exit(100);

	wmstate_atom = XInternAtom(display, "WM_STATE", False);
	state_proto_atom = XInternAtom(display, "_WWM_SET_STATE", False);

	select_eventsG(wmstate_handle, root_w, SubstructureNotifyMask);

	register_sendevent_handler(wmstate_handle, root_w,
		UnmapNotify, e_root_unmap);
}
