/*
**	Copyright (c) 1998, 1999 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions are
**	met:
**
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include "deco.h"
#include "focus.h"
#include "colormap.h"
#include "wmstate.h"
#include "event.h"
#include "resource.h"

#ifndef lint
static char sccsid[] = "@(#)wwm.c	1.100 2/8/99";
#endif

#ifdef DEBUG_EVENTS
#ifndef lint
static char sccsid2[] = "@(#)+DEBUG_EVENTS";
#endif
#endif


typedef struct
{
	unsigned mask;
	XWindowChanges v;
} configureT;

/* currently decorated clients are never sticky */

typedef struct
{
	Window w;
	int is_sticky, overrides, decorate;
	int is_deco; /* so restack it despite override_redirect */

	/* queued requests */
	int do_query, do_map, close_down;
	configureT configure;
} *childT;

typedef struct
{
	Window client;
	int width, height;
	int client_border; /* width */
	decoT deco;

	/* queued requests */
	int close_down;
	configureT configure;
} *clientT;


static int grabbed;
static jmp_buf grab_jmp_buf;

static Display *display;
static int screen;
static Window root_w, master_w /* None if destroyed */;
static Atom is_sticky_atom;

static int n_children;
static childT *children;

static int n_clients;
static clientT *clients;

static int n_decos;
static decoT *decos;

static void *main_handle; /* for the event module */


static void mem_error(void)
{
	fprintf(stderr, "out of memory\n");
	exit(1);
}


static void new_child(Window w, int overrides)
{
	int i;
	childT c;

	n_children++;
	if ((children = realloc(children,
		n_children * sizeof *children)) == NULL)
		mem_error();
	if ((c = malloc(sizeof *c)) == NULL)
		mem_error();

	if (n_children > 1)
		memmove(&children[1], &children[0],
			(n_children - 1) * sizeof *children);

#ifdef DEBUG_EVENTS
	printf("+WwmChild h=%p w=%#lx\n", c, (unsigned long)w);
#endif

	children[0] = c;

	c->w = w;
	c->do_query = 1;
	c->do_map = c->close_down = 0;
	c->is_sticky = -1; /* unknown */
	c->overrides = overrides;
	c->decorate = 0; /* unknown */
	c->configure.mask = 0;

	for (i = 0; i < n_clients &&
		w != deco_toplevel(clients[i]->deco); i++);
	c->is_deco = i < n_clients;
}


static childT lookup_child(Window w)
{
	int i;

	for (i = 0; i < n_children && children[i]->w != w; i++);
	if (i >= n_children)
		abort();

	return children[i];
}


static void drop_child(Window w)
{
	int i;

	for (i = 0; i < n_children && children[i]->w != w; i++);
	if (i >= n_children)
		return; /* w is no child */

	children[i]->close_down = 1;
}


static void remove_childG(childT c)
{
	int i;

	for (i = 0; i < n_children && children[i] != c; i++);
	if (i >= n_children)
		abort();

	unregister_event_handlersG(children[i]);

#ifdef DEBUG_EVENTS
	printf("-WwmChild h=%p w=%#lx\n", children[i],
		(unsigned long)children[i]->w);
#endif
	free(children[i]);

	n_children--;
	if (n_children > i)
		memmove(&children[i], &children[i + 1],
			(n_children - i) * sizeof *children);
}


static void child_above(Window w, Window above)
{
	int i;
	childT c;

	for (i = 0; i < n_children && children[i]->w != w; i++);
	if (i >= n_children)
		abort();

	c = children[i];

	n_children--;
	if (n_children > i)
		memmove(&children[i], &children[i + 1],
			(n_children - i) * sizeof *children);

	for (i = 0; i < n_children && children[i]->w != above; i++);

	if (i < n_children)
		memmove(&children[i + 1], &children[i],
			(n_children - i) * sizeof *children);
	n_children++;
	children[i] = c;
}


static int need_to_restack(void)
{
	int i;

	/* search for a movable window */
	for (i = 0; i < n_children; i++)
		if (children[i]->is_sticky == 0 &&
			(children[i]->overrides == 0 || children[i]->is_deco))
			/* this window is certainly movable */
			break;

	/* search for a sticky window below it */
	for (; i < n_children; i++)
		if (children[i]->is_sticky == 1)
			/* this window is known to be sticky */
			break;

	/* found something? */
	return i < n_children;
}


static void restack_windowsG(void)
{
	int i, j, k, n;
	Window *list;

	/* find the lowest sticky window */
	for (i = j = n_children; i-- > 0;)
		if (children[i]->is_sticky == 1)
		{
			j = i;
			break;
		}

	/* none? */
	if (j >= n_children)
		return;

	/* count movable windows above it (plus itself) */
	for (i = 0, n = 1; i < j; i++)
		if (children[i]->is_sticky == 0 &&
			(children[i]->overrides == 0 || children[i]->is_deco))
			n++;

	/* nothing to restack? */
	if (n < 2)
		return;

	if ((list = malloc(n * sizeof *list)) == NULL)
		mem_error();

	/* collect windows */
	k = 0;
	list[k++] = children[j]->w;
	for (i = 0; k < n; i++)
		if (children[i]->is_sticky == 0 &&
			(children[i]->overrides == 0 || children[i]->is_deco))
			list[k++] = children[i]->w;

	XRestackWindows(display, list, n);
	free(list);
}


/*
** sibling may be None;
** returns the window to stay below;
** may return None for "top"
*/
static Window lowest_sticky(Window sibling, int stack_mode)
{
	int i;

	/* bottom is never restricted */
	if (sibling == None && stack_mode == Below)
		return n_children == 0 ? None : children[n_children - 1]->w;

	for (i = n_children; i-- > 0;)
		if (children[i]->is_sticky)
			return children[i]->w;
		else if (children[i]->w == sibling)
			if (stack_mode == Below)
				return children[i]->w;
			else
				return i == 0 ? None : children[i - 1]->w;

	return None;
}


static void drop_client(Window w)
{
	int i;

	for (i = 0; i < n_clients; i++)
		if (clients[i]->client == w)
			break;
	if (i >= n_clients)
		return; /* not related to a client */

	clients[i]->close_down = 1;
}


static void remove_clientG(clientT c)
{
	int i;

	for (i = 0; i < n_clients && clients[i] != c; i++);
	if (i >= n_clients)
		abort();

	unregister_event_handlersG(clients[i]);
#ifdef DEBUG_EVENTS
	printf("-WwmClient h=%p w=%#lx\n", clients[i],
		(unsigned long)clients[i]->client);
#endif
	free(clients[i]);

	n_clients--;
	if (n_clients > i)
		memmove(&clients[i], &clients[i + 1],
			(n_clients - i) * sizeof *clients);
}


static void merge_configure(configureT *destP, configureT *srcP)
{
	unsigned mask, src_mask;

	mask = destP->mask;
	src_mask = srcP->mask;

	if (src_mask & CWX)
		destP->v.x = srcP->v.x,
		mask |= CWX;

	if (src_mask & CWY)
		destP->v.y = srcP->v.y,
		mask |= CWY;

	if (src_mask & CWWidth)
		destP->v.width = srcP->v.width,
		mask |= CWWidth;

	if (src_mask & CWHeight)
		destP->v.height = srcP->v.height,
		mask |= CWHeight;

	if (src_mask & CWBorderWidth)
		destP->v.border_width = srcP->v.border_width,
		mask |= CWBorderWidth;

	if (src_mask & (CWSibling | CWStackMode))
	{
		destP->v.sibling = srcP->v.sibling;
		destP->v.stack_mode = srcP->v.stack_mode;
		mask &= ~(CWSibling | CWStackMode);
		mask |= src_mask & (CWSibling | CWStackMode);
	}

	destP->mask = mask;
}


static void try_configure_window(Window w, configureT *configureP)
{
	int i;

	for (i = 0; i < n_clients; i++)
		if (clients[i]->client == w)
		{
			merge_configure(&clients[i]->configure, configureP);
			return;
		}
		else if (deco_toplevel(clients[i]->deco) == w)
		{
			translate_coords(&configureP->v);
			merge_configure(&clients[i]->configure, configureP);
			return;
		}

	for (i = 0; i < n_children; i++)
		if (children[i]->w == w)
		{
			merge_configure(&children[i]->configure, configureP);
			return;
		}
}


/*ARGSUSED*/
static void e_destroy(void *handle, XEvent *evP)
{
	Window w;

	w = evP->xdestroywindow.window;

	if (w == master_w)
	{
		master_w = None; /* signal main() */
		return;
	}

	drop_client(w);
	drop_child(w);
}


static void e_child_property(void *handle, XEvent *evP)
{
	childT c;

	c = handle;

	if (evP->xproperty.atom == is_sticky_atom)
		c->is_sticky = evP->xproperty.state == PropertyNewValue;
}


/*ARGSUSED*/
static void e_clip_configure_request(void *handle, XEvent *evP)
{
	configureT c;

	c.mask = evP->xconfigurerequest.value_mask;

	c.v.x = evP->xconfigurerequest.x;
	c.v.y = evP->xconfigurerequest.y;
	c.v.width = evP->xconfigurerequest.width;
	c.v.height = evP->xconfigurerequest.height;
	c.v.border_width = evP->xconfigurerequest.border_width;
	c.v.sibling = evP->xconfigurerequest.above;
	c.v.stack_mode = evP->xconfigurerequest.detail;

	try_configure_window(evP->xconfigurerequest.window, &c);
}


/*ARGSUSED*/
static void e_root_configure_request(void *handle, XEvent *evP)
{
	configureT c;

	c.mask = evP->xconfigurerequest.value_mask;

	c.v.x = evP->xconfigurerequest.x;
	c.v.y = evP->xconfigurerequest.y;
	c.v.width = evP->xconfigurerequest.width;
	c.v.height = evP->xconfigurerequest.height;
	c.v.border_width = evP->xconfigurerequest.border_width;
	c.v.sibling = evP->xconfigurerequest.above;
	c.v.stack_mode = evP->xconfigurerequest.detail;

	try_configure_window(evP->xconfigurerequest.window, &c);
}


/*ARGSUSED*/
static void e_root_create(void *handle, XEvent *evP)
{
	new_child(evP->xcreatewindow.window,
		evP->xcreatewindow.override_redirect);
}


/*ARGSUSED*/
static void e_root_map_request(void *handle, XEvent *evP)
{
	lookup_child(evP->xmaprequest.window)->do_map = 1;
}


/*ARGSUSED*/
static void e_root_circulate_request(void *handle, XEvent *evP)
{
	configureT c;

	c.mask = CWStackMode;

	switch (evP->xcirculaterequest.place)
	{
	case PlaceOnTop:
		c.v.stack_mode = Above;
		break;

	case PlaceOnBottom:
		c.v.stack_mode = Below;
		break;

	default:
		abort();
	}

	try_configure_window(evP->xcirculaterequest.window, &c);
}


/*ARGSUSED*/
static void e_root_configure_notify(void *handle, XEvent *evP)
{
	child_above(evP->xconfigure.window, evP->xconfigure.above);
}


/*ARGSUSED*/
static void e_root_circulate_notify(void *handle, XEvent *evP)
{
	child_above(evP->xcirculate.window,
		evP->xcirculate.place == PlaceOnTop &&
			n_children > 0 ? children[0]->w : None);
}


/*ARGSUSED*/
static void e_root_reparent(void *handle, XEvent *evP)
{
	if (evP->xreparent.parent == root_w)
	{
		Window w;
		int i;

		w = evP->xreparent.window;

		for (i = 0; i < n_children && children[i]->w != w; i++);
		if (i >= n_children)
			new_child(w, evP->xreparent.override_redirect);
	}
	else
		drop_child(evP->xreparent.window);
}


static int resize_kludgeG(Window w, XSizeHints *hints, long *maskP)
{
	XClassHint class;
	char name[200], class_name[200];

	if (!XGetClassHint(display, w, &class))
		return 0;

	if (class.res_name == NULL || class.res_class == NULL ||
		strlen(class.res_name) > 100 || strlen(class.res_class) > 100)
	{
		if (class.res_name != NULL)
			XFree(class.res_name);
		if (class.res_class != NULL)
			XFree(class.res_class);
		return 0;
	}

	strcpy(name, "wwm.client.");
	strcat(name, class.res_name);
	strcat(name, ".resizeKludge");

	strcpy(class_name, "Wwm.Client.");
	strcat(class_name, class.res_class);
	strcat(class_name, ".ResizeKludge");

	XFree(class.res_name);
	XFree(class.res_class);

	if (!resource_lookup_bool(name, class_name, False))
		return 0;

	return XGetWMNormalHints(display, w, hints, maskP);
}


static void decorate_and_mapG(childT child, XWindowAttributes *attrP,
	int new_map)
{
	decoT deco;
	Window below;
	int x, y, width, height, border;
	clientT c;

	/* protect the window as soon as possible */
	wmstate_will_decorateG(child->w);

	border = attrP->border_width;
	x = attrP->x + border;
	y = attrP->y + border;
	width = attrP->width;
	height = attrP->height;

	/* XXX workaround for obsolete clients like netscape and xv */
	if (new_map)
	{
		XSizeHints *hints;
		long mask;

		hints = XAllocSizeHints();
		if (resize_kludgeG(child->w, hints, &mask))
		{
			mask &= hints->flags;

			if (mask & (USPosition | PPosition))
			{
				x = hints->x;
				y = hints->y;
			}

			if (mask & (USSize | PSize))
			{
				width = hints->width;
				height = hints->height;
				XResizeWindow(display, child->w,
					width, height);
			}
		}
		XFree(hints);
	}

	deco = decorate_windowG(child->w, root_w, x, y, width, height);

	n_decos++;
	if ((decos = realloc(decos, n_decos * sizeof *decos)) == NULL)
		mem_error();
	decos[n_decos - 1] = deco;

	/* stay below all sticky windows (request top) */
	below = lowest_sticky(None, Above);
	if (below != None)
	{
		Window list[2];

		list[0] = below;
		list[1] = deco_toplevel(deco);
		XRestackWindows(display, list, 2);
	}

	n_clients++;
	if ((clients = realloc(clients, n_clients * sizeof *clients)) == NULL)
		mem_error();
	if ((c = malloc(sizeof *c)) == NULL)
		mem_error();

#ifdef DEBUG_EVENTS
	printf("+WwmClient h=%p w=%#lx deco=%p\n",
		c, (unsigned long)child->w, deco);
#endif

	clients[n_clients - 1] = c;

	c->client = child->w;
	c->close_down = 0;
	c->width = width;
	c->height = height;
	c->client_border = border;
	c->deco = deco;

	colormap_new_clientG(deco_toplevel(deco), child->w);

	/* move configure request with the window before dropping child */
	memcpy(&c->configure, &child->configure, sizeof c->configure);
	child->configure.mask = 0;

	select_eventsG(c, get_clip_window(deco),
		SubstructureRedirectMask | SubstructureNotifyMask);
	register_event_handler(c, get_clip_window(deco), DestroyNotify,
		e_destroy);
	register_event_handler(c, get_clip_window(deco), ConfigureRequest,
		e_clip_configure_request);
	register_sendevent_handler(c, get_clip_window(deco), ConfigureRequest,
		e_clip_configure_request);

	/* stop processing enter events before child->w is mapped */
	select_eventsG(lookup_child(child->w), child->w, NoEventMask);

	{
		Window list[2];

		list[0] = child->w;
		list[1] = deco_toplevel(deco);

		/* put the toplevel below the client to maintain
		   its stacking position; the client is still a sibling */
		XRestackWindows(display, list, 2);
	}

	XReparentWindow(display, child->w, get_clip_window(deco),
		-border, -border);

	wmstate_reparent(child->w, get_clip_window(deco));

	XMapWindow(display, child->w);
	XMapWindow(display, deco_toplevel(deco));
}


static void drop_deco(int i)
{
	int j;

	for (j = 0; j < n_clients; j++)
		if (!clients[j]->close_down && clients[j]->deco == decos[i])
		{
			clients[j]->close_down = 1;
			break;
		}

	n_decos--;
	if (n_decos > i)
		memmove(&decos[i], &decos[i + 1],
			(n_decos - i) * sizeof *decos);
}


static void query_childG(childT c, XWindowAttributes *attrP)
{
	c->do_query = 0;

	if (!c->is_deco)
		colormap_new_windowG(c->w);

	select_eventsG(c, c->w, PropertyChangeMask);
	register_event_handler(c, c->w, PropertyNotify, e_child_property);

	{
		Atom type_atom;
		int format;
		unsigned long n_items, bytes_after;
		unsigned char *prop;

		if (XGetWindowProperty(display, c->w, is_sticky_atom,
			0L /* offset */, 0L /* length */, False /* delete */,
			AnyPropertyType, &type_atom, &format, &n_items,
			&bytes_after, &prop) != Success)
			exit(1);

		if (format == 0)
			c->is_sticky = 0;
		else
		{
			c->is_sticky = 1;
			XFree(prop);
		}
	}

	{
		XWindowAttributes attr;
		int deco;

		if (!XGetWindowAttributes(display, c->w, &attr))
			exit(1);
		c->overrides = attr.override_redirect;

		deco = !attr.override_redirect &&
			attr.class == InputOutput && !c->is_deco;
		c->decorate = deco;

		if (attrP != NULL)
			memcpy(attrP, &attr, sizeof *attrP);

		if (c->decorate)
			wmstate_new_windowG(c->w,
				attr.map_state == IsViewable ?
					NormalState : WithdrawnState);
	}
}


static void map_childG(childT c)
{
	c->do_map = 0;

	if (c->decorate)
	{
		XWindowAttributes attr;

		if (!XGetWindowAttributes(display, c->w, &attr))
			exit(1);
		decorate_and_mapG(c, &attr, !wmstate_was_iconic(c->w));
	}
	else
		XMapWindow(display, c->w);
}


static void do_configureG(clientT c)
{
	int border;
	XWindowChanges v;
	unsigned mask;

	if ((c->configure.mask & CWWidth) &&
		(c->configure.v.width < 1 ||
		c->configure.v.width > 65535))
		c->configure.mask &= ~CWWidth;
	if ((c->configure.mask & CWHeight) &&
		(c->configure.v.height < 1 ||
		c->configure.v.height > 65535))
		c->configure.mask &= ~CWHeight;

	border = c->client_border;

	mask = c->configure.mask &
		(CWWidth | CWHeight | CWBorderWidth);

	/* some of the parameters might be garbage */
	v.width = c->configure.v.width;
	v.height = c->configure.v.height;
	v.border_width = c->configure.v.border_width;

	if (mask & CWBorderWidth)
	{
		border = c->client_border = c->configure.v.border_width;

		v.x = -border;
		v.y = -border;
#ifdef DEBUG_EVENTS
		printf("h=%p serial=%d w=%#lx mask=%#x\n",
			c, NextRequest(display), (unsigned long)c->client,
			CWX | CWY | mask);
#endif
		XConfigureWindow(display, c->client, CWX | CWY | mask, &v);
	}
	else if (mask)
	{
#ifdef DEBUG_EVENTS
		printf("h=%p serial=%d w=%#lx mask=%#x\n",
			c, NextRequest(display),
			(unsigned long)c->client, mask);
#endif
		XConfigureWindow(display, c->client, mask, &v);
	}

	if (c->configure.mask & CWWidth)
		c->width = c->configure.v.width;
	if (c->configure.mask & CWHeight)
		c->height = c->configure.v.height;

	mask = c->configure.mask & (CWX | CWY | CWWidth | CWHeight);

	/* some of the parameters might be garbage */
	v.x = c->configure.v.x;
	v.y = c->configure.v.y;
	v.width = c->configure.v.width;
	v.height = c->configure.v.height;

	/* handle restacking requests */
	if ((c->configure.mask & CWStackMode) &&
		(c->configure.v.stack_mode == Above ||
		c->configure.v.stack_mode == Below))
	{
		Window below;

		if (c->configure.mask & CWSibling)
			below = lowest_sticky(c->configure.v.sibling,
				c->configure.v.stack_mode);
		else
			below = lowest_sticky(None, c->configure.v.stack_mode);

		if (below == None)
		{
			v.stack_mode = Above;
			mask |= CWStackMode;
		}
		else if (below != deco_toplevel(c->deco))
		{
			v.stack_mode = Below;
			v.sibling = below;
			mask |= CWStackMode | CWSibling;
		}
	}

	if (mask)
		configure_decoG(c->deco, mask, &v);

	c->configure.mask = 0;

	{
		XEvent e;
		Window child_dummy;
		int x, y;

		if (!XTranslateCoordinates(display, c->client,
			root_w, 0, 0, &x, &y, &child_dummy))
			exit(1);

		memset(&e, 0, sizeof e);
		e.type = ConfigureNotify;
		e.xconfigure.event = c->client;
		e.xconfigure.window = c->client;
		e.xconfigure.x = x;
		e.xconfigure.y = y;
		e.xconfigure.width = c->width;
		e.xconfigure.height = c->height;
		e.xconfigure.border_width = border;
		e.xconfigure.above = None;
		e.xconfigure.override_redirect = False;

		if (!XSendEvent(display, c->client, False,
			StructureNotifyMask, &e))
			exit(1);
	}
}


/*ARGSUSED*/
static int redirect_fail_handler(Display *d, XErrorEvent *error)
{
	fprintf(stderr, "There is already a window manager running.\n");
	exit(2);

	/* dummy */
	return 0;
}


static void grab_server(void)
{
	if (!grabbed)
		longjmp(grab_jmp_buf, 1);
}


static int process_pending_request(void)
{
	int i;

	for (i = 0; i < n_children; i++)
		if (children[i]->close_down)
		{
			grab_server();
			remove_childG(children[i]);
			return 1;
		}

	for (i = 0; i < n_clients; i++)
		if (clients[i]->close_down)
		{
			grab_server();
			remove_clientG(clients[i]);
			return 1;
		}

	for (i = 0; i < n_children; i++)
		if (children[i]->do_query)
		{
			grab_server();
			query_childG(children[i], NULL);
			return 1;
		}

	if (need_to_restack())
	{
		grab_server();
		restack_windowsG();
		/* XXX need the ReparentNotify */
		XSync(display, False);
		return 1;
	}

	for (i = 0; i < n_children; i++)
		if (children[i]->do_map)
		{
			grab_server();
			map_childG(children[i]);
			return 1;
		}

	for (i = 0; i < n_clients; i++)
		if (clients[i]->configure.mask)
		{
			grab_server();
			do_configureG(clients[i]);
			return 1;
		}

	for (i = 0; i < n_children; i++)
		if (children[i]->configure.mask)
		{
			grab_server();
			/* XXX check .sibling */
#ifdef DEBUG_EVENTS
			printf("h=%p serial=%d w=%#lx mask=%#x\n", children[i],
				NextRequest(display),
				(unsigned long)children[i]->w,
				children[i]->configure.mask);
#endif
			XConfigureWindow(display, children[i]->w,
				children[i]->configure.mask,
				&children[i]->configure.v);
			children[i]->configure.mask = 0;
			return 1;
		}

	if (wmstate_has_work())
	{
		grab_server();
		wmstate_do_workG();
		return 1;
	}

	if (colormap_has_work())
	{
		grab_server();
		colormap_do_workG();
		return 1;
	}

	if (focus_has_work())
	{
		grab_server();
		focus_do_workG();
		return 1;
	}

	for (i = 0; i < n_decos; i++)
		if (deco_has_work(decos[i]))
		{
			grab_server();
			if (!deco_do_workG(decos[i]))
				drop_deco(i);
			return 1;
		}

	return 0;
}


/* ----- */


void reparent_clientG(Window w)
{
	int i;
	int border;
	int x, y;
	Window child;

	for (i = 0; i < n_clients && clients[i]->client != w; i++);
	if (i >= n_clients)
		return; /* already gone */

	border = clients[i]->client_border;

	if (!XTranslateCoordinates(display, w, root_w,
		-border, -border, &x, &y, &child))
		exit(1);

	XUnmapWindow(display, w);
	XReparentWindow(display, w, root_w, x, y);
	/* the ReparentNotify will destroy the client entry */

	{
		Window list[2];

		list[0] = deco_toplevel(clients[i]->deco);
		list[1] = clients[i]->client;

		/* put the client below the toplevel to maintain
		   its stacking position; the client is now a sibling */
		if (list[0] != None)
			XRestackWindows(display, list, 2);
	}

	wmstate_reparent(w, root_w);
}


void iconify_windowG(Window w)
{
	reparent_clientG(w);
}


void deiconify_windowG(Window w)
{
	map_childG(lookup_child(w));
}


int main(int argc, char **argv)
{
	if (argc != 1)
		exit(10);

	n_children = 0;
	if ((children = malloc(sizeof *children)) == NULL)
		mem_error();

	n_clients = 0;
	if ((clients = malloc(sizeof *clients)) == NULL)
		mem_error();

	n_decos = 0;
	if ((decos = malloc(sizeof *decos)) == NULL)
		mem_error();

	if ((main_handle = malloc(1)) == NULL)
		mem_error();

	display = XOpenDisplay(NULL);
	if (display == NULL)
	{
		fprintf(stderr, "can't open display \"%s\"\n",
			XDisplayName(NULL));
		exit(1);
	}

	screen = DefaultScreen(display);
	root_w = DefaultRootWindow(display);

	event_init(display);

	{
		int (*old_handler)(Display *display, XErrorEvent *error);

		XSync(display, False);
		old_handler = XSetErrorHandler(redirect_fail_handler);

		/* doesn't require a grab */
		select_eventsG(main_handle, root_w,
			SubstructureRedirectMask | SubstructureNotifyMask);
		XSync(display, False);

		XSetErrorHandler(old_handler);
	}

	register_event_handler(main_handle, root_w, ConfigureRequest,
		e_root_configure_request);
	register_sendevent_handler(main_handle, root_w, ConfigureRequest,
		e_root_configure_request);
	register_event_handler(main_handle, root_w, DestroyNotify, e_destroy);
	register_event_handler(main_handle, root_w, CreateNotify,
		e_root_create);
	register_event_handler(main_handle, root_w, MapRequest,
		e_root_map_request);
	register_event_handler(main_handle, root_w, CirculateRequest,
		e_root_circulate_request);
	register_sendevent_handler(main_handle, root_w, CirculateRequest,
		e_root_circulate_request);
	register_event_handler(main_handle, root_w, ConfigureNotify,
		e_root_configure_notify);
	register_event_handler(main_handle, root_w, CirculateNotify,
		e_root_circulate_notify);
	register_event_handler(main_handle, root_w, ReparentNotify,
		e_root_reparent);

	XGrabServer(display);
	XSync(display, True); /* send Grab, drop all stale events */

	/* create the master window after the root has SubstructureNotifyMask
	   so we can't miss the master window's disappearing */
	{
		XTextProperty prop, *propP;
		char *name;
		XSetWindowAttributes attr;
		static XClassHint hint = { "wwm", "Wwm" };

		attr.override_redirect = True;
		master_w = XCreateWindow(display, root_w, -10, -10, 5, 5, 0,
			CopyFromParent, CopyFromParent, CopyFromParent,
			CWOverrideRedirect, &attr);
		created_with_event_maskG(main_handle, master_w, NoEventMask);

		name = "wwm";
		if (XStringListToTextProperty(&name, 1, &prop))
			propP = &prop;
		else
			propP = NULL;

		XSetWMProperties(display, master_w, propP, propP, argv, argc,
			NULL, NULL, &hint);

		if (propP != NULL)
			XFree(propP->value);
	}

	/* now we can destroy argc/argv */
	resource_init(display, &argc, argv);

	is_sticky_atom = XInternAtom(display, "_WWM_IS_STICKY", False);

	deco_init(display, screen, root_w);
	focus_initG(display, root_w);
	colormap_initG(display, screen, root_w);
	wmstate_initG(display, root_w);

	{
		Window dummy_root, dummy_parent;
		Window *list;
		unsigned n;

		if (!XQueryTree(display, root_w, &dummy_root, &dummy_parent,
			&list, &n))
			exit(1);

		if (n)
		{
			int i;

			n_children = n;
			if ((children = realloc(children,
				n_children * sizeof *children)) == NULL)
				mem_error();

			for (i = 0; i < n; i++)
			{
				childT c;

				if ((c = malloc(sizeof *c)) == NULL)
					mem_error();

				children[i] = c;

				c->w = list[n - 1 - i];
				c->do_query = 1;
				c->do_map = c->close_down = 0;
				c->is_sticky = -1; /* unknown */
				c->overrides = -1; /* unknown */
				c->decorate = 0; /* unknown */
				c->is_deco = 0;
				c->configure.mask = 0;

#ifdef DEBUG_EVENTS
				printf("+WwmChild h=%p w=%#lx\n",
					c, (unsigned long)c->w);
#endif

				colormap_new_windowG(c->w);
			}

			XFree(list);

			for (i = 0; i < n; i++)
			{
				childT c;
				XWindowAttributes attr;

				c = children[i];
				query_childG(c, &attr); /* tells wmstate */

				if (c->decorate &&
					attr.map_state == IsViewable)
					decorate_and_mapG(c, &attr, 0);
			}
		}
	}

	XUngrabServer(display);
	XFlush(display); /* send Ungrab quickly */

	grabbed = 0;
	while (master_w != None)
	{
		XEvent ev;

		if (XEventsQueued(display, QueuedAlready))
		{
			if (grabbed)
			{
				XUngrabServer(display);
				XFlush(display); /* send Ungrab quickly */
				grabbed = 0;
#ifdef DEBUG_EVENTS
				puts("Ungrab");
#endif
			}

			XNextEvent(display, &ev);
			handle_event(&ev);
			continue;
		}

		if (setjmp(grab_jmp_buf))
		{
			/* via grab_server() and longjmp() */
			XGrabServer(display);
			XSync(display, False); /* send Grab, gather events */
			grabbed = 1;
#ifdef DEBUG_EVENTS
			puts("Grab");
#endif
			continue;
		}

		if (process_pending_request())
			continue;

		if (grabbed)
		{
			XUngrabServer(display);
			XFlush(display); /* send Ungrab quickly */
			grabbed = 0;
#ifdef DEBUG_EVENTS
			puts("Ungrab");
#endif
		}

#ifdef DEBUG_EVENTS
		/* will block now */
		fflush(stdout);
#endif

		XNextEvent(display, &ev);
		handle_event(&ev);
	}

	return 0;
}
