/*
**	Copyright (c) 1998, 1999 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions are
**	met:
**
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/cursorfont.h>
#include <X11/Xmu/WinUtil.h>

#include "query_pointer.h"

#ifndef lint
static char sccsid[] = "@(#)xclose.c	1.6 2/8/99";
#endif


#define SPACE 15


static void kill_window(Display *display, Window w, Time t)
{
	Atom del_atom, proto_atom, *protocols;
	int i, n;

	del_atom = XInternAtom(display, "WM_DELETE_WINDOW",
		True /* if exists */);
	proto_atom = XInternAtom(display, "WM_PROTOCOLS",
		False /* if exists */);

	if (del_atom != None && proto_atom != None &&
		XGetWMProtocols(display, w, &protocols, &n) && n > 0)
	{
		for (i = 0; i < n && protocols[i] != del_atom; i++);
		if (i < n)
		{
			XEvent e;

			memset(&e, 0, sizeof e);
			e.xclient.type = ClientMessage;
			e.xclient.window = w;
			e.xclient.message_type = proto_atom;
			e.xclient.format = 32;
			e.xclient.data.l[0] = del_atom;
			e.xclient.data.l[1] = t;
			XSendEvent(display, w, False /* propagate */,
				NoEventMask, &e);
			return;
		}
	}

	XDestroyWindow(display, w);
}


int main(int argc, char **argv)
{
	Display *display;
	int screen;
	Window root_w, dummy_w, target_w;
	Window query_w, yes_w, no_w;
	XWindowAttributes attr;
	XSetWindowAttributes v;
	XFontStruct *font;
	Colormap cmap;
	XColor screen_c, exact_c;
	int x, y;
	int yes_width, no_width;
	int w, h;
	XGCValues gc_v;
	GC text_gc, top_gc, bottom_gc;

	if (argc != 1)
	{
		fprintf(stderr, "usage: %s\n", argv[0]);
		exit(10);
	}

	display = XOpenDisplay(NULL);
	if (display == NULL)
	{
		fprintf(stderr, "%s: can't open display %s\n",
			argv[0], XDisplayName(NULL));
		exit(1);
	}

	screen = DefaultScreen(display);
	root_w = DefaultRootWindow(display);
	cmap = DefaultColormap(display, screen);

	if ((target_w = query_pointer(display, NULL, NULL)) == None)
	{
		XBell(display, 0);
		XCloseDisplay(display);
		exit(1);
	}

	target_w = XmuClientWindow(display, target_w);
	if (!XGetWindowAttributes(display, target_w, &attr))
		exit(1);

	if (!XTranslateCoordinates(display, target_w, root_w,
		attr.width / 2, attr.height / 2, &x, &y, &dummy_w))
		exit(1);

	font = XLoadQueryFont(display,
		"-*-courier-bold-r-*-*-*-140-*-*-*-*-*-*");
	if (font == NULL)
		font = XLoadQueryFont(display, "fixed");
	if (font == NULL)
		exit(1);

	yes_width = XTextWidth(font, "yes", strlen("yes"));
	no_width = XTextWidth(font, "no", strlen("no"));

	w = (yes_width > no_width ? yes_width : no_width) + 2 * SPACE;
	h = font->ascent + font->descent + 2 * SPACE;

	v.border_pixel = BlackPixel(display, screen);
	v.event_mask = StructureNotifyMask | SubstructureNotifyMask;
	query_w = XCreateWindow(display, root_w,
		x - w, y - h / 2, 2 * w, h, 1,
		CopyFromParent /* depth */, InputOutput,
		CopyFromParent /* visual */,
		CWBorderPixel | CWEventMask, &v);

	if (XAllocNamedColor(display, cmap, "#c0c0c0", &screen_c, &exact_c))
		v.background_pixel = screen_c.pixel;
	else
		v.background_pixel = WhitePixel(display, screen);
	v.event_mask = ButtonPressMask | ExposureMask;
	v.cursor = XCreateFontCursor(display, XC_pirate);
	yes_w = XCreateWindow(display, query_w, 0, 0, w, h, 0,
		CopyFromParent /* depth */, InputOutput,
		CopyFromParent /* visual */,
		CWBackPixel | CWEventMask | CWCursor, &v);
	no_w = XCreateWindow(display, query_w, w, 0, w, h, 0,
		CopyFromParent /* depth */, InputOutput,
		CopyFromParent /* visual */,
		CWBackPixel | CWEventMask, &v);

	gc_v.foreground = BlackPixel(display, screen);
	gc_v.font = font->fid;
	text_gc = XCreateGC(display, query_w, GCForeground | GCFont, &gc_v);

	if (XAllocNamedColor(display, cmap, "#e8e8e8", &screen_c, &exact_c))
		gc_v.foreground = screen_c.pixel;
	else
		gc_v.foreground = WhitePixel(display, screen);
	top_gc = XCreateGC(display, query_w, GCForeground, &gc_v);

	if (XAllocNamedColor(display, cmap, "#909090", &screen_c, &exact_c))
		gc_v.foreground = screen_c.pixel;
	else
		gc_v.foreground = BlackPixel(display, screen);
	bottom_gc = XCreateGC(display, query_w, GCForeground, &gc_v);

	XStoreName(display, query_w, "close?");
	XMapWindow(display, yes_w);
	XMapWindow(display, no_w);
	XMapWindow(display, query_w);

	for (;;)
	{
		XEvent ev;

		XNextEvent(display, &ev);
		switch (ev.type)
		{
		case DestroyNotify:
			break;

		case Expose:
			XDrawLine(display, ev.xexpose.window, bottom_gc,
				1, h - 1, w - 1, h - 1);
			XDrawLine(display, ev.xexpose.window, bottom_gc,
				w - 1, h - 2, w - 1, 0);
			XDrawLine(display, ev.xexpose.window, top_gc,
				0, 0, 0, h - 1);
			XDrawLine(display, ev.xexpose.window, top_gc,
				1, 0, w - 2, 0);

			if (ev.xexpose.window != no_w)
				XDrawString(display, ev.xexpose.window,
					text_gc, (w - yes_width) / 2,
					font->ascent + SPACE,
					"yes", strlen("yes"));
			else
				XDrawString(display, ev.xexpose.window,
					text_gc, (w - no_width) / 2,
					font->ascent + SPACE,
					"no", strlen("no"));
			continue;

		case ButtonPress:
			if (ev.xbutton.window == yes_w)
				kill_window(display, target_w,
					ev.xbutton.time);
			break;

		default:
			continue;
		}

		XCloseDisplay(display);
		return 0;
	}
}
