/*
**	Copyright (c) 1998, 1999 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions are
**	met:
**
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/cursorfont.h>
#include <X11/Xmu/WinUtil.h>
#include <X11/extensions/shape.h>

#include "query_pointer.h"

#ifndef lint
static char sccsid[] = "@(#)xmove.c	1.12 2/8/99";
#endif


#define EDGE_DELTA 20


static Display *display;
static int screen;

static int space_width, space_height; /* may be negative */
static int space_x, space_y;

static unsigned char bits[] = { 0x38, 0x1c, 0x0e, 0x07, 0x23, 0x31 };


static Region rect_region(int x, int y, int width, int height)
{
	XRectangle rect;
	Region r;

	rect.x = x;
	rect.y = y;
	rect.width = width;
	rect.height = height;

	r = XCreateRegion();
	XUnionRectWithRegion(&rect, r, r);

	return r;
}


static Region get_shape(Window w, int *dxP, int *dyP, Region *full_shapeP,
	unsigned *borderP)
{
	Window dummy_w;
	Region r, r2;
	XRectangle *list;
	int x, y, n, ordering, dummy_x, dummy_y;
	int b_shaped, c_shaped;
	unsigned width, height, dummy_width, dummy_height;
	unsigned border, depth;

	if (!XGetGeometry(display, w, &dummy_w, &x, &y,
		&width, &height, &border, &depth))
		exit(1);

	if (!XShapeQueryExtents(display, w, &b_shaped, &dummy_x, &dummy_y,
		&dummy_width, &dummy_height, &c_shaped, &dummy_x, &dummy_y,
		&dummy_width, &dummy_height))
		exit(1);

	if (b_shaped)
	{
		if ((list = XShapeGetRectangles(display, w, ShapeBounding,
			&n, &ordering)) == NULL || n < 1)
			exit(1);

		r = XCreateRegion();
		while (n--)
			XUnionRectWithRegion(&list[n], r, r);
		XFree(list);
	}
	else
		r = rect_region(-border, -border, width + 2 * border,
			height + 2 * border);

	r2 = XCreateRegion();
	XUnionRegion(r2, r, r2);
	XOffsetRegion(r2, *dxP, *dyP);
	*full_shapeP = r2;

	r2 = rect_region(0, 0, width, height);
	XIntersectRegion(r, r2, r);
	XDestroyRegion(r2);

	XOffsetRegion(r, *dxP, *dyP);

	*dxP -= x + border;
	*dyP -= y + border;

	if (borderP != NULL)
		*borderP = border;

	return r;
}


static void pointer_moves(int *xP, int *yP, int new_x, int new_y)
{
	int x, y;

	x = *xP - space_x;
	y = *yP - space_y;

	new_x -= space_x;
	new_y -= space_y;

	if (x >= 0 && new_x < 0 && new_x > -EDGE_DELTA)
		new_x = 0;

	if (x <= space_width && new_x > space_width &&
		new_x < space_width + EDGE_DELTA)
		new_x = space_width;

	if (y >= 0 && new_y < 0 && new_y > -EDGE_DELTA)
		new_y = 0;

	if (y <= space_height && new_y > space_height &&
		new_y < space_height + EDGE_DELTA)
		new_y = space_height;

	*xP = new_x + space_x;
	*yP = new_y + space_y;
}


int main(int argc, char **argv)
{
	Window dummy_w, root_w, frame_w, target_w, w, border_w;
	Region frame_shape, target_shape, border_shape;
	XRectangle border_rect, frame_rect;
	XSetWindowAttributes v;
	int root_x, root_y, x, y, dx, dy;
	int root_width, root_height;
	unsigned border;

	if (argc != 1)
	{
		fprintf(stderr, "usage: %s\n", argv[0]);
		exit(10);
	}

	display = XOpenDisplay(NULL);
	if (display == NULL)
	{
		fprintf(stderr, "%s: can't open display %s\n",
			argv[0], XDisplayName(NULL));
		exit(1);
	}

	screen = DefaultScreen(display);
	root_w = DefaultRootWindow(display);

	root_width = WidthOfScreen(ScreenOfDisplay(display, screen));
	root_height = HeightOfScreen(ScreenOfDisplay(display, screen));

	{
		int event_base, error_base;

		if (!XShapeQueryExtension(display, &event_base, &error_base))
			exit(1);
	}

	if (XGrabPointer(display, root_w, False /* owner_events */,
		ButtonPressMask | PointerMotionMask,
		GrabModeAsync, GrabModeAsync, None /* confine_to */,
		XCreateFontCursor(display, XC_fleur), CurrentTime) !=
		GrabSuccess)
		exit(1);

	if ((frame_w = query_pointer(display, &root_x, &root_y)) == None)
	{
		XBell(display, 0);
		XCloseDisplay(display);
		exit(1);
	}

	target_w = XmuClientWindow(display, frame_w);
	XSelectInput(display, target_w, StructureNotifyMask);

	/* current absolute position, ignoring any border */
	if (!XTranslateCoordinates(display, target_w, root_w,
		0, 0, &x, &y, &dummy_w))
		exit(1);

	w = target_w;
	dx = dy = 0;
	target_shape = get_shape(w, &dx, &dy, &frame_shape, &border);

	while (w != frame_w)
	{
		Window parent_w, *children;
		Region r;
		unsigned n;

		if (!XQueryTree(display, w, &dummy_w, &parent_w,
			&children, &n) || parent_w == root_w)
			exit(1);
		if (n)
			XFree(children);

		XDestroyRegion(frame_shape);

		w = parent_w;
		r = get_shape(w, &dx, &dy, &frame_shape, NULL);
		XIntersectRegion(target_shape, r, target_shape);
		XDestroyRegion(r);

		border = 0;
	}

	border_shape = XCreateRegion();
	XSubtractRegion(frame_shape, target_shape, border_shape);

	if (XEmptyRegion(border_shape))
		if (XEmptyRegion(target_shape))
			exit(1);
		else
		{
			XClipBox(target_shape, &frame_rect);
			XDestroyRegion(border_shape);

			border_shape = rect_region(frame_rect.x - 1,
				frame_rect.y - 1, frame_rect.width + 2,
				frame_rect.height + 2);
			XSubtractRegion(border_shape, target_shape,
				border_shape);

			XClipBox(border_shape, &border_rect);
		}
	else
		XClipBox(border_shape, &frame_rect);

	XDestroyRegion(frame_shape);
	XDestroyRegion(target_shape);

	XClipBox(border_shape, &border_rect);

	v.background_pixmap = XCreatePixmapFromBitmapData(display, root_w,
		(void *)bits, 6, 6, BlackPixel(display, screen),
		WhitePixel(display, screen), DefaultDepth(display, screen));
	v.override_redirect = True;
	v.save_under = True;
	v.event_mask = StructureNotifyMask;
	border_w = XCreateWindow(display, root_w, x + border_rect.x,
		y + border_rect.y, border_rect.width, border_rect.height,
		0 /* border_width */, CopyFromParent /* depth */,
		InputOutput, CopyFromParent /* visual */,
		CWBackPixmap | CWOverrideRedirect | CWSaveUnder | CWEventMask,
		&v);

	XShapeCombineRegion(display, border_w, ShapeBounding,
		-border_rect.x, -border_rect.y, border_shape, ShapeSet);

	XMapWindow(display, border_w);

	/* root_x/_y = position of pointer */
	dx = root_x - x;
	dy = root_y - y;

	space_width = root_width - frame_rect.width;
	space_height = root_height - frame_rect.height;
	space_x = dx - frame_rect.x;
	space_y = dy - frame_rect.y;

	{
		/* the pointer might have moved before the grab */
		int r_x, r_y, dummy_x, dummy_y;
		unsigned mask;

		if (XQueryPointer(display, root_w, &dummy_w, &dummy_w,
			&r_x, &r_y, &dummy_x, &dummy_y, &mask))
			pointer_moves(&root_x, &root_y, r_x, r_y);
	}

	for (;;)
	{
		XEvent ev;

		if (!XEventsQueued(display, QueuedAlready))
		{
			XMoveWindow(display, border_w,
				root_x - dx + border_rect.x,
				root_y - dy + border_rect.y);
			XSync(display, False /* discard */);
		}

		XNextEvent(display, &ev);
		if (ev.type == DestroyNotify)
			break;

		if (ev.type == ButtonPress)
		{
			if (ev.xbutton.button == Button1 &&
				ev.xbutton.same_screen)
			{
				pointer_moves(&root_x, &root_y,
					ev.xbutton.x_root, ev.xbutton.y_root);

				XMoveWindow(display, target_w,
					root_x - dx - border,
					root_y - dy - border);
			}
			break;
		}

		if (ev.type == MotionNotify && ev.xmotion.same_screen)
			pointer_moves(&root_x, &root_y,
				ev.xmotion.x_root, ev.xmotion.y_root);
	}

	XCloseDisplay(display);
	return 0;
}
