/*
**	Copyright (c) 1998, 1999 by Joerg Czeranski
**	All rights reserved.
**
**	Redistribution and use in source and binary forms, with or without
**	modification, are permitted provided that the following conditions are
**	met:
**
**	1. Redistributions of source code must retain the above copyright
**	   notice, this list of conditions and the following disclaimer.
**	2. Redistributions in binary form must reproduce the above copyright
**	   notice, this list of conditions and the following disclaimer in the
**	   documentation and/or other materials provided with the distribution.
**	3. The name of the author may not be used to endorse or promote
**	   products derived from this software without specific prior written
**	   permission.
**
**	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
**	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
**	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
**	DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
**	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
**	POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/cursorfont.h>
#include <X11/Xmu/WinUtil.h>
#include <X11/extensions/shape.h>

#include "query_pointer.h"

#ifndef lint
static char sccsid[] = "@(#)xresize.c	1.9 2/8/99";
#endif


static Display *display;
static int screen;

static unsigned grab_event_mask;

static int target_x, target_y, target_width, target_height; /* original */
static int catch_xl, catch_xr, catch_yt, catch_yb;

static unsigned char bits[] = { 0x38, 0x1c, 0x0e, 0x07, 0x23, 0x31 };

static int cursors[3][3] =
{
	{ XC_top_left_corner, XC_top_side, XC_top_right_corner },
	{ XC_left_side, XC_fleur, XC_right_side },
	{ XC_bottom_left_corner, XC_bottom_side, XC_bottom_right_corner }
};


static Region rect_region(int x, int y, int width, int height)
{
	XRectangle rect;
	Region r;

	rect.x = x;
	rect.y = y;
	rect.width = width;
	rect.height = height;

	r = XCreateRegion();
	XUnionRectWithRegion(&rect, r, r);

	return r;
}


static Region rect_subtract_region(XRectangle *outer_rect,
	XRectangle *inner_rect)
{
	Region r_outer, r_inner;

	r_outer = XCreateRegion();
	r_inner = XCreateRegion();

	XUnionRectWithRegion(outer_rect, r_outer, r_outer);
	XUnionRectWithRegion(inner_rect, r_inner, r_inner);

	XSubtractRegion(r_outer, r_inner, r_outer);
	XDestroyRegion(r_inner);

	return r_outer;
}


static Region get_shape(Window w, int *dxP, int *dyP, Region *full_shapeP,
	unsigned *borderP)
{
	Window dummy_w;
	Region r, r2;
	int x, y;
	unsigned width, height;
	unsigned border, depth;

	if (!XGetGeometry(display, w, &dummy_w, &x, &y,
		&width, &height, &border, &depth))
		exit(1);

	r = rect_region(-border, -border, width + 2 * border,
		height + 2 * border);

	r2 = XCreateRegion();
	XUnionRegion(r2, r, r2);
	XOffsetRegion(r2, *dxP, *dyP);
	*full_shapeP = r2;

	r2 = rect_region(0, 0, width, height);
	XIntersectRegion(r, r2, r);
	XDestroyRegion(r2);

	XOffsetRegion(r, *dxP, *dyP);

	*dxP -= x + border;
	*dyP -= y + border;

	if (borderP != NULL)
		*borderP = border;

	return r;
}


static void pointer_moves(int *xP, int *yP, int new_x, int new_y)
{
	int old_cursor, new_cursor;

	old_cursor = cursors[1 + catch_yb - catch_yt][1 + catch_xr - catch_xl];

	if (new_x < target_x || new_x == target_x && !catch_xr)
	{
		catch_xl = 1;
		catch_xr = 0;
	}
	else if (new_x >= target_x + target_width ||
		new_x == target_x + target_width - 1 && !catch_xl)
	{
		catch_xl = 0;
		catch_xr = 1;
	}

	if (new_y < target_y || new_y == target_y && !catch_yb)
	{
		catch_yt = 1;
		catch_yb = 0;
	}
	else if (new_y >= target_y + target_height ||
		new_y == target_y + target_height - 1 && !catch_yt)
	{
		catch_yt = 0;
		catch_yb = 1;
	}

	new_cursor = cursors[1 + catch_yb - catch_yt][1 + catch_xr - catch_xl];

	if (new_cursor != old_cursor)
	{
		Cursor cursor;

		cursor = XCreateFontCursor(display, new_cursor);
		XChangeActivePointerGrab(display, grab_event_mask,
			cursor, CurrentTime);
		XFreeCursor(display, cursor);
	}

	*xP = new_x;
	*yP = new_y;
}


int main(int argc, char **argv)
{
	Window dummy_w, root_w, frame_w, target_w, w, border_w;
	Region frame_shape, target_shape, border_shape;
	XRectangle border_rect, frame_rect, target_rect;
	XSetWindowAttributes v;
	int root_x, root_y, reference_x, reference_y, dx, dy;
	int width_delta, height_delta;
	unsigned border;
	Cursor cursor;

	if (argc != 1)
	{
		fprintf(stderr, "usage: %s\n", argv[0]);
		exit(10);
	}

	display = XOpenDisplay(NULL);
	if (display == NULL)
	{
		fprintf(stderr, "%s: can't open display %s\n",
			argv[0], XDisplayName(NULL));
		exit(1);
	}

	screen = DefaultScreen(display);
	root_w = DefaultRootWindow(display);

	{
		int event_base, error_base;

		if (!XShapeQueryExtension(display, &event_base, &error_base))
			exit(1);
	}

	grab_event_mask = ButtonPressMask | PointerMotionMask;

	cursor = XCreateFontCursor(display, cursors[1][1]);
	if (XGrabPointer(display, root_w, False /* owner_events */,
		grab_event_mask, GrabModeAsync, GrabModeAsync,
		None /* confine_to */, cursor, CurrentTime) != GrabSuccess)
		exit(1);
	XFreeCursor(display, cursor);

	if ((frame_w = query_pointer(display, &root_x, &root_y)) == None)
	{
		XBell(display, 0);
		XCloseDisplay(display);
		exit(1);
	}

	target_w = XmuClientWindow(display, frame_w);
	XSelectInput(display, target_w, StructureNotifyMask);

	/* current absolute position, ignoring any border */
	if (!XTranslateCoordinates(display, target_w, root_w,
		0, 0, &reference_x, &reference_y, &dummy_w))
		exit(1);

	w = target_w;
	dx = dy = 0;
	target_shape = get_shape(w, &dx, &dy, &frame_shape, &border);

	while (w != frame_w)
	{
		Window parent_w, *children;
		Region r;
		unsigned n;

		if (!XQueryTree(display, w, &dummy_w, &parent_w,
			&children, &n) || parent_w == root_w)
			exit(1);
		if (n)
			XFree(children);

		XDestroyRegion(frame_shape);

		w = parent_w;
		r = get_shape(w, &dx, &dy, &frame_shape, NULL);
		XIntersectRegion(target_shape, r, target_shape);
		XDestroyRegion(r);

		border = 0;
	}

	if (XEmptyRegion(target_shape))
		exit(1);

	XClipBox(target_shape, &target_rect);

	border_shape = XCreateRegion();
	XUnionRegion(frame_shape, target_shape, border_shape);
	XClipBox(border_shape, &frame_rect);
	XDestroyRegion(border_shape);

	border_shape = rect_region(target_rect.x - 1,
		target_rect.y - 1, target_rect.width + 2,
		target_rect.height + 2);
	XUnionRegion(border_shape, frame_shape, border_shape);
	XClipBox(border_shape, &border_rect);
	XDestroyRegion(border_shape);

	XDestroyRegion(frame_shape);
	XDestroyRegion(target_shape);

	width_delta = border_rect.width - target_rect.width;
	height_delta = border_rect.height - target_rect.height;

	v.background_pixmap = XCreatePixmapFromBitmapData(display, root_w,
		(void *)bits, 6, 6, BlackPixel(display, screen),
		WhitePixel(display, screen), DefaultDepth(display, screen));
	v.override_redirect = True;
	v.save_under = True;
	v.event_mask = StructureNotifyMask;
	border_w = XCreateWindow(display, root_w, reference_x + border_rect.x,
		reference_y + border_rect.y, border_rect.width,
		border_rect.height, 0 /* border_width */,
		CopyFromParent /* depth */, InputOutput,
		CopyFromParent /* visual */,
		CWBackPixmap | CWOverrideRedirect | CWSaveUnder | CWEventMask,
		&v);

	border_shape = rect_subtract_region(&border_rect, &target_rect);
	XShapeCombineRegion(display, border_w, ShapeBounding,
		-border_rect.x, -border_rect.y, border_shape, ShapeSet);
	XDestroyRegion(border_shape);

	XMapWindow(display, border_w);

	/* root_x/_y = position of pointer */
	dx = root_x - reference_x;
	dy = root_y - reference_y;

	target_x = target_rect.x + reference_x;
	target_y = target_rect.y + reference_y;
	target_width = target_rect.width;
	target_height = target_rect.height;

	catch_xl = catch_xr = catch_yt = catch_yb = 0;

	{
		/* the pointer might have moved before the grab */
		int r_x, r_y, dummy_x, dummy_y;
		unsigned mask;

		if (XQueryPointer(display, root_w, &dummy_w, &dummy_w,
			&r_x, &r_y, &dummy_x, &dummy_y, &mask))
			pointer_moves(&root_x, &root_y, r_x, r_y);
	}

	for (;;)
	{
		XEvent ev;

		if (!XEventsQueued(display, QueuedAlready))
		{
			int x, y, width, height;
			int max_width, max_height;

			/* x/y of target */
			x = catch_xl ? root_x : target_x;
			y = catch_yt ? root_y : target_y;

			width = catch_xl ? target_x + target_width - root_x :
				catch_xr ? root_x + 1 - target_x :
				target_width;
			height = catch_yt ? target_y + target_height - root_y :
				catch_yb ? root_y + 1 - target_y :
				target_height;

			max_width = border_rect.width;
			max_height = border_rect.height;

			target_rect.width = width;
			target_rect.height = height;

			border_rect.width = width + width_delta;
			border_rect.height = height + height_delta;

			if (max_width < border_rect.width ||
				max_height < border_rect.height)
			{
				if (max_width < border_rect.width)
					max_width = border_rect.width;
				if (max_height < border_rect.height)
					max_height = border_rect.height;

				XResizeWindow(display, border_w,
					max_width, max_height);
			}

			border_shape = rect_subtract_region(&border_rect,
				&target_rect);
			XShapeCombineRegion(display, border_w, ShapeBounding,
				-border_rect.x, -border_rect.y, border_shape,
				ShapeSet);
			XDestroyRegion(border_shape);

			XMoveResizeWindow(display, border_w,
				x + border_rect.x - target_rect.x,
				y + border_rect.y - target_rect.y,
				border_rect.width, border_rect.height);

			XSync(display, False /* discard */);
		}

		XNextEvent(display, &ev);
		if (ev.type == DestroyNotify)
			break;

		if (ev.type == ButtonPress)
		{
			if (ev.xbutton.button == Button1 &&
				ev.xbutton.same_screen)
			{
				int x, y, width, height;

				pointer_moves(&root_x, &root_y,
					ev.xbutton.x_root, ev.xbutton.y_root);

				/* x/y of target */
				x = catch_xl ? root_x : target_x;
				y = catch_yt ? root_y : target_y;

				width = catch_xl ? target_x + target_width -
						root_x :
					catch_xr ? root_x + 1 - target_x :
					target_width;
				height = catch_yt ? target_y + target_height -
						root_y :
					catch_yb ? root_y + 1 - target_y :
					target_height;

				target_rect.width = width;
				target_rect.height = height;

				border_rect.width = width + width_delta;
				border_rect.height = height + height_delta;

				XMoveResizeWindow(display, target_w,
					x - target_rect.x - border,
					y - target_rect.y - border,
					target_rect.width, target_rect.height);
			}
			break;
		}

		if (ev.type == MotionNotify && ev.xmotion.same_screen)
			pointer_moves(&root_x, &root_y,
				ev.xmotion.x_root, ev.xmotion.y_root);
	}

	XCloseDisplay(display);
	return 0;
}
